package metrics

import (
	"context"
	"math/rand"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/aws/aws-sdk-go/service/cloudwatch/cloudwatchiface"
)

const (
	cloudwatchRequestMax  = 20
	cloudwatchChannelSize = 100
)

type CW struct {
	client cloudwatchiface.CloudWatchAPI

	namespace string
	interval  time.Duration
	onError   func(error)

	buffer chan *message
}

func NewCW(client cloudwatchiface.CloudWatchAPI, namespace string,
	interval time.Duration, onError func(error)) *CW {

	c := &CW{
		client:    client,
		namespace: namespace,
		interval:  interval,
		onError:   onError,
		buffer:    make(chan *message, cloudwatchChannelSize),
	}
	return c
}

// Enqueue queues a metric datum for submission to CloudWatch, returning
// whether the datum was successfully enqueued.
func (c *CW) Enqueue(datum *cloudwatch.MetricDatum) (enqueued bool) {
	select {
	case c.buffer <- &message{datum: datum}:
		return true
	default:
		return false
	}
}

// Flush blocks until all previously-enqueued metrics have been submitted, or
// until the provided Context expires.
func (c *CW) Flush(ctx context.Context) error {
	return flush(ctx, c.buffer)
}

// Run drains the metric queue, returning only once the provided Context has
// expired.
//
// It writes the metrics out to CloudWatch in three cases. First, it does
// periodic writes based on the interval provided. Second, when the resulting
// request would be the maximum size supported by the backend. Third, in
// response to a call to Flush.
func (c *CW) Run(ctx context.Context) error {
	ctx, cancel := context.WithCancel(ctx)
	defer cancel()

	prng := rand.New(rand.NewSource(time.Now().UnixNano()))

	s := &sender{
		sleep:   func(i int) time.Duration { return jitterDuration(i, c.interval, prng.Int63n) },
		buffer:  &untypedSlice{data: make([]interface{}, cloudwatchRequestMax)},
		ch:      c.buffer,
		send:    c.send,
		onError: c.onError,
	}

	return s.run(ctx)
}

func (c *CW) send(ctx context.Context, ds dataSlice) error {
	ctx, cancel := context.WithTimeout(ctx, 10*time.Second)
	defer cancel()

	// TODO: use a dataSlice backed by []*cloudwatch.MetricDatum
	data := ds.(*untypedSlice).data

	req := &cloudwatch.PutMetricDataInput{
		Namespace:  aws.String(c.namespace),
		MetricData: make([]*cloudwatch.MetricDatum, len(data)),
	}
	for i := range data {
		req.MetricData[i], data[i] = data[i].(*cloudwatch.MetricDatum), nil
	}

	_, err := c.client.PutMetricDataWithContext(ctx, req)
	return err
}
