package metrics

import (
	"context"
	"math/rand"
	"time"

	"code.justin.tv/video/mwsclient"
)

const (
	// https://w.amazon.com/index.php/Monitoring/Documentation/MonitoringWebService/PutMetricDataForAggregation#Maximum_number_of_metrics_per_request
	mwsRequestMax  = 10000
	mwsChannelSize = 100
)

type MWS struct {
	client *mwsclient.MWS

	namespace string
	metadata  *mwsclient.MetricMetadata
	interval  time.Duration
	onError   func(error)

	buffer chan *message
}

func NewMWS(client *mwsclient.MWS, meta *mwsclient.MetricMetadata,
	interval time.Duration, onError func(error)) *MWS {

	c := &MWS{
		client:    client,
		namespace: "Amazon/Schema/Service",
		metadata:  meta,
		interval:  interval,
		onError:   onError,
		buffer:    make(chan *message, mwsChannelSize),
	}
	return c
}

// Enqueue queues a metric datum for submission to MWS, returning whether the
// datum was successfully enqueued.
func (c *MWS) Enqueue(datum *mwsclient.Metric) (enqueued bool) {
	select {
	case c.buffer <- &message{datum: datum}:
		return true
	default:
		return false
	}
}

// Flush blocks until all previously-enqueued metrics have been submitted, or
// until the provided Context expires.
func (c *MWS) Flush(ctx context.Context) error {
	return flush(ctx, c.buffer)
}

// Run drains the metric queue, returning only once the provided Context has
// expired.
//
// It writes the metrics out to MWS in three cases. First, it does periodic
// writes based on the interval provided. Second, when the resulting request
// would be the maximum size supported by the backend. Third, in response to a
// call to Flush.
func (c *MWS) Run(ctx context.Context) error {
	ctx, cancel := context.WithCancel(ctx)
	defer cancel()

	prng := rand.New(rand.NewSource(time.Now().UnixNano()))

	s := &sender{
		sleep:   func(i int) time.Duration { return jitterDuration(i, c.interval, prng.Int63n) },
		buffer:  &untypedSlice{data: make([]interface{}, mwsRequestMax)},
		ch:      c.buffer,
		send:    c.send,
		onError: c.onError,
	}

	return s.run(ctx)
}

func (c *MWS) send(ctx context.Context, ds dataSlice) error {
	ctx, cancel := context.WithTimeout(ctx, 10*time.Second)
	defer cancel()

	// TODO: use a dataSlice backed by []*mwsclient.Metric
	data := ds.(*untypedSlice).data

	report := &mwsclient.MetricReport{
		Metadata:  c.metadata,
		Namespace: c.namespace,
		Metrics:   make([]*mwsclient.Metric, len(data)),
	}
	for i := range data {
		report.Metrics[i], data[i] = data[i].(*mwsclient.Metric), nil
	}

	req := &mwsclient.PutMetricDataForAggregationInput{MetricReports: []*mwsclient.MetricReport{report}}
	_, err := c.client.PutMetricDataForAggregation(ctx, req)
	return err
}
