package stream

import "context"

type StreamProcessor interface {
	ProcessStream(ctx context.Context, out Enqueuer, in Dequeuer) error
}

type Stream interface {
	EnqueueFinisher
	Dequeuer
}

type Dequeuer interface {
	Dequeue(ctx context.Context) (interface{}, bool)
}

type Enqueuer interface {
	Enqueue(ctx context.Context, val interface{}) bool
}

type EnqueueFinisher interface {
	Enqueuer
	Finish()
}

type UntypedStream chan interface{}

func (s UntypedStream) Enqueue(ctx context.Context, val interface{}) bool {
	select {
	case s <- val:
		return true
	default:
		// consult the context only if the immediate enqueue fails
		select {
		case s <- val:
			return true
		case <-ctx.Done():
			return false
		}
	}
}

func (s UntypedStream) Finish() { close(s) }

func (s UntypedStream) Dequeue(ctx context.Context) (interface{}, bool) {
	select {
	case val, ok := <-s:
		return val, ok
	default:
		// consult the context only if the immediate dequeue fails
		select {
		case val, ok := <-s:
			return val, ok
		case <-ctx.Done():
			return nil, false
		}
	}
}

func FiniteStream(v ...interface{}) Stream {
	s := make(UntypedStream, len(v))
	for _, val := range v {
		s.Enqueue(context.Background(), val)
	}
	s.Finish()
	return s
}

type FanoutEnqueue []Enqueuer

func (f FanoutEnqueue) Enqueue(ctx context.Context, val interface{}) bool {
	allOK := true
	for _, e := range f {
		ok := e.Enqueue(ctx, val)
		if !ok {
			allOK = false
		}
	}
	return allOK
}
