package persistent

import (
	"context"
	"io/ioutil"
	"net/http"
	"os"
	"path"
	"strings"
	"testing"
	"time"

	"code.justin.tv/release/trace"
	"code.justin.tv/release/trace/analysis/tx"
	"code.justin.tv/release/trace/api"
)

func TestOpenDB(t *testing.T) {
	dir, err := ioutil.TempDir("", "db")
	if err != nil {
		t.Fatalf("TempDir; err = %q", err)
	}
	defer os.RemoveAll(dir)

	db, err := OpenDB(dir)
	if err != nil {
		t.Fatalf("OpenDB; err = %q", err)
	}
	defer db.Close()

	err = db.WriteTransaction(&api.Transaction{
		TransactionId: []uint64{0, 1},
		Client: &api.Service{
			Name: "foo",
		},
	})
	if err != nil {
		t.Errorf("WriteTransaction; err = %q", err)
	}

	err = db.CompactAll()
	if err != nil {
		t.Errorf("CompactAll; err = %q", err)
	}

	infos, err := ioutil.ReadDir(dir)
	if err != nil {
		t.Fatalf("ReadDir; err = %q", err)
	}

	for _, info := range infos {
		if strings.HasSuffix(info.Name(), ".log") && info.Size() > 0 {
			t.Errorf("CompactAll; log, size = %q, %d", info.Name(), info.Size())
		}
	}
}

func TestOpenDBFail(t *testing.T) {
	db, err := OpenDB(os.Args[0])
	if err == nil {
		t.Errorf("OpenDB(os.Args[0]); err == nil")
	}
	if db != nil {
		t.Errorf("OpenDB(os.Args[0]); db != nil")
		db.Close()
	}
}

func TestNewFilesystemReportIndex(t *testing.T) {
	start := time.Now()

	dir, err := ioutil.TempDir("", "db")
	if err != nil {
		t.Fatalf("TempDir; err = %q", err)
	}
	defer os.RemoveAll(dir)

	txid := [2]uint64{0, 1}

	{
		db, err := OpenDB(path.Join(dir, "1234", "db"))
		if err != nil {
			t.Fatalf("OpenDB; err = %q", err)
		}

		err = db.WriteTransaction(&api.Transaction{
			TransactionId: txid[:],
			Client: &api.Service{
				Name: "foo",
			},
		})
		if err != nil {
			t.Errorf("WriteTransaction; err = %q", err)
		}
		err = db.CompactAll()
		if err != nil {
			t.Errorf("CompactAll; err = %q", err)
		}
		db.Close()
	}

	fs, err := NewFilesystemReportIndex(http.Dir(dir))
	if err != nil {
		t.Fatalf("NewFilesystemReportIndex; err = %q", err)
	}

	ctx := context.Background()

	reports, err := fs.List(ctx)
	if err != nil {
		t.Fatalf("List; err = %q", err)
	}

	if have, want := len(reports), 1; have != want {
		t.Fatalf("len(reports); %d != %d", have, want)
	}

	rep := reports[0]
	if have, want := rep.ID(), "1234"; have != want {
		t.Errorf("rep.ID(); %q != %q", have, want)
	}

	if have, now := rep.ModTime(), time.Now(); have.Before(start.Add(-2*time.Second)) || have.After(now.Add(2*time.Second)) {
		t.Errorf("rep.ModTime(); %s not within [%s, %s]", have, start, now)
	}

	if have, want := tx.TransactionID(txid).String(), "0000000000000000"+"0100000000000000"; have != want {
		t.Errorf("tx.TransactionID(txid).String();\n%q\n!=\n%q\n", have, want)
	}

	tx, err := rep.Pluck(ctx, trace.ParseID("0000000000000000"+"0100000000000000"))
	if err != nil {
		t.Fatalf("rep.Pluck(00000000000000000100000000000000); err = %q", err)
	}
	if have, want := tx.Client.Name, "foo"; have != want {
		t.Errorf("tx.Client.Name; %q != %q", have, want)
	}
}
