package persistent

import (
	"context"
	"fmt"
	"io"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
)

// NewS3ObjectReader provides random access to data stored in S3 using the
// io.ReaderAt interface.
func NewS3ObjectReader(ctx context.Context, s3client s3iface.S3API, bucket, key string, size int64) (*S3ObjectReader, error) {
	rd := &S3ObjectReader{
		ctx:    ctx,
		s3:     s3client,
		bucket: bucket,
		key:    key,
		size:   size,
	}

	return rd, nil
}

type S3ObjectReader struct {
	ctx context.Context
	s3  s3iface.S3API

	bucket string
	key    string

	size int64
}

var _ io.Closer = (*S3ObjectReader)(nil)
var _ io.ReaderAt = (*S3ObjectReader)(nil)

func (rd *S3ObjectReader) Size() int64 {
	return rd.size
}

func (rd *S3ObjectReader) Close() error {
	return nil
}

func (rd *S3ObjectReader) ReadAt(p []byte, off int64) (n int, err error) {
	if remainder := rd.size - off; remainder < int64(len(p)) {
		if remainder < 0 {
			remainder = 0
		}
		p = p[:int(remainder)]
	}

	if len(p) == 0 {
		return 0, io.ErrShortBuffer
	}

	resp, err := rd.s3.GetObjectWithContext(rd.ctx, &s3.GetObjectInput{
		Bucket: aws.String(rd.bucket),
		Key:    aws.String(rd.key),
		Range:  aws.String(fmt.Sprintf("bytes=%d-%d", off, off-1+int64(len(p)))),
	})
	if err != nil {
		return 0, err
	}
	defer func() {
		cerr := resp.Body.Close()
		if err == nil {
			err = cerr
		}
	}()

	return io.ReadFull(resp.Body, p)
}
