package persistent

import (
	"archive/zip"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"sort"
	"strings"

	"github.com/syndtr/goleveldb/leveldb/storage"
)

// NewZipLDBStorage opens an uncompressed zip archive for use as read-only
// storage for a LevelDB database. The archive's files must be stored
// uncompressed. The archive is assumed to have the given size in bytes.
func NewZipLDBStorage(r io.ReaderAt, size int64) (storage.Storage, error) {
	zr, err := zip.NewReader(r, size)
	if err != nil {
		return nil, err
	}

	z := &zipStorage{
		r:    r,
		size: size,
		zr:   zr,
	}
	return z, nil
}

var (
	errReadOnly      = errors.New("storage is read-only")
	errSeekUnderflow = errors.New("seek to negative offset")
	errInvalidSeek   = errors.New("invalid seek")
)

type zipStorage struct {
	r    io.ReaderAt
	size int64
	zr   *zip.Reader
}

var _ storage.Storage = (*zipStorage)(nil)

// GetMeta returns the file descriptor for the database's metadata file.
func (z *zipStorage) GetMeta() (storage.FileDesc, error) {
	// TODO: read CURRENT file
	fds, err := z.List(storage.TypeManifest)
	if err != nil {
		return storage.FileDesc{}, err
	}
	if len(fds) != 1 {
		return storage.FileDesc{}, fmt.Errorf("need exactly one meta: %q", fds)
	}
	return fds[0], nil
}

// List returns file descriptors that match the given file types.
func (z *zipStorage) List(ft storage.FileType) ([]storage.FileDesc, error) {
	var fds []storage.FileDesc
	for _, f := range z.zr.File {
		fd, ok := fsParseName(f.Name)
		if !ok || fd.Type&ft == 0 {
			continue
		}
		fds = append(fds, fd)
	}
	return fds, nil
}

// Open returns a read-only handle for the file with the given file
// descriptor.
func (z *zipStorage) Open(fd storage.FileDesc) (storage.Reader, error) {
	name := fd.String()
	for _, f := range z.zr.File {
		if f.Name == name {
			return newZipStorageFile(z, f)
		}
	}
	return nil, os.ErrNotExist
}

func (z *zipStorage) Log(str string) {}
func (z *zipStorage) Close() error   { return nil }
func (z *zipStorage) Lock() (storage.Locker, error) {
	return fnLocker(func() {}), nil
}

func (z *zipStorage) SetMeta(fd storage.FileDesc) error                  { return errReadOnly }
func (z *zipStorage) Create(fd storage.FileDesc) (storage.Writer, error) { return nil, errReadOnly }
func (z *zipStorage) Remove(fd storage.FileDesc) error                   { return errReadOnly }
func (z *zipStorage) Rename(oldfd, newfd storage.FileDesc) error         { return errReadOnly }

type zipStorageFile struct {
	stor *zipStorage
	f    *zip.File

	dataOffset      int64
	intraFileOffset int64
}

// newZipStorageFile opens a file in a zip archive for read access. The file
// must be stored uncompressed.
func newZipStorageFile(stor *zipStorage, f *zip.File) (*zipStorageFile, error) {
	if f.Method != zip.Store {
		return nil, fmt.Errorf("zip contents must be uncompressed, but storage method is %d", f.Method)
	}

	dataOffset, err := f.DataOffset()
	if err != nil {
		return nil, err
	}

	zf := &zipStorageFile{stor: stor, f: f, dataOffset: dataOffset}

	return zf, nil
}

func (zf *zipStorageFile) Read(p []byte) (int, error) {
	n, err := zf.ReadAt(p, zf.intraFileOffset)
	zf.Seek(int64(n), io.SeekCurrent)
	return n, err
}

// Seek sets the offset for the next Read or Write to offset, interpreted
// according to whence: io.SeekStart means relative to the start of the file,
// io.SeekCurrent means relative to the current offset, and io.SeekEnd means
// relative to the end. Seek returns the new offset relative to the start of
// the file and an error, if any.
//
// Seeking to an offset before the start of the file is an error. Seeking to
// any positive offset is legal, but the behavior of subsequent I/O operations
// on the underlying object is implementation-dependent.
func (zf *zipStorageFile) Seek(offset int64, whence int) (int64, error) {
	newOffset := zf.intraFileOffset
	switch whence {
	case io.SeekStart:
		newOffset = offset
	case io.SeekEnd:
		newOffset = int64(zf.f.CompressedSize64) + offset
	case io.SeekCurrent:
		newOffset += offset
	default:
		return zf.intraFileOffset, errInvalidSeek
	}

	if newOffset < 0 {
		return zf.intraFileOffset, errSeekUnderflow
	}

	zf.intraFileOffset = newOffset
	return zf.intraFileOffset, nil
}

func (zf *zipStorageFile) Close() error {
	return nil
}

func (zf *zipStorageFile) ReadAt(p []byte, off int64) (int, error) {
	readStart := zf.dataOffset + off
	readEnd := readStart + int64(len(p))

	if fileEnd := zf.dataOffset + int64(zf.f.CompressedSize64); readEnd > fileEnd {
		readEnd = fileEnd
	}
	p = p[:readEnd-readStart]
	if len(p) == 0 {
		return 0, io.EOF
	}

	n, err := zf.stor.r.ReadAt(p, readStart)
	return n, err
}

type countingWriter struct {
	w  io.Writer
	nn int64
}

func (cw *countingWriter) Write(p []byte) (int, error) {
	n, err := cw.w.Write(p)
	cw.nn += int64(n)
	return n, err
}

type ZipWriter struct {
	stor storage.Storage
}

// NewZipWriter returns a ZipWriter prepared to write out the contents of a
// LevelDB storage backend in an uncompressed zip format.
func NewZipWriter(stor storage.Storage) *ZipWriter {
	return &ZipWriter{stor: stor}
}

func (l *ZipWriter) WriteTo(w io.Writer) (int64, error) {
	cw := &countingWriter{w: w}
	zw := zip.NewWriter(cw)

	fds, err := l.stor.List(storage.TypeManifest | storage.TypeJournal | storage.TypeTable)
	if err != nil {
		return cw.nn, err
	}
	sort.Slice(fds, func(i, j int) bool {
		fi, fj := fds[i], fds[j]
		if fi.Type != fj.Type {
			return fi.Type < fj.Type
		}
		return fi.Num < fj.Num
	})

	mfd, err := l.stor.GetMeta()
	if err != nil {
		return cw.nn, err
	}
	err = l.addFile(zw, "CURRENT", ioutil.NopCloser(strings.NewReader(fmt.Sprintf("%s\n", mfd.String()))))
	if err != nil {
		return cw.nn, err
	}

	for _, fd := range fds {
		rd, err := l.stor.Open(fd)
		if err != nil {
			return cw.nn, err
		}
		err = l.addFile(zw, fd.String(), rd)
		if err != nil {
			return cw.nn, err
		}
	}

	err = zw.Close()
	if err != nil {
		return cw.nn, err
	}
	return cw.nn, nil
}

func (_ *ZipWriter) addFile(zw *zip.Writer, name string, rd io.ReadCloser) (err error) {
	defer func() {
		cerr := rd.Close()
		if err == nil {
			err = cerr
		}
	}()

	// zip.Store allows the zip archive's contents to be accessed as an io.ReaderAt
	w, err := zw.CreateHeader(&zip.FileHeader{Name: name, Method: zip.Store})
	if err != nil {
		return err
	}

	_, err = io.Copy(w, rd)
	if err != nil {
		return err
	}

	return nil
}
