package loadroutes

import (
	"bytes"
	"compress/gzip"
	"fmt"
	"reflect"
	"strings"
	"testing"

	"code.justin.tv/common/alvin/internal/httpproto"
	"code.justin.tv/common/alvin/internal/httpproto/google_api"
	"github.com/golang/protobuf/proto"
	"github.com/golang/protobuf/protoc-gen-go/descriptor"

	_ "code.justin.tv/common/alvin/internal/httpproto/loadroutes/loadtestproto"
)

func withExtension(t *testing.T, opts *descriptor.MethodOptions, val proto.Message) *descriptor.MethodOptions {
	b, err := proto.Marshal(val)
	if err != nil {
		t.Fatalf("proto.Marshal(%#v); err = %v", val, err)
	}

	b = append(proto.EncodeVarint(uint64(len(b))), b...)
	b = append(proto.EncodeVarint(uint64(eHttp.Field<<3|proto.WireBytes)), b...)
	proto.SetRawExtension(opts, eHttp.Field, b)
	return opts
}

func TestFromServiceDescriptor(t *testing.T) {

	for _, tt := range []struct {
		protoPackage string
		serviceDesc  *descriptor.ServiceDescriptorProto
		specs        []*httpproto.MethodSpec
	}{
		{
			protoPackage: "",
			serviceDesc:  nil,
			specs:        []*httpproto.MethodSpec(nil),
		},
		{
			protoPackage: "test.package",
			serviceDesc: &descriptor.ServiceDescriptorProto{
				Method: []*descriptor.MethodDescriptorProto{nil},
			},
			specs: []*httpproto.MethodSpec(nil),
		},
		{
			protoPackage: "test.package",
			serviceDesc: &descriptor.ServiceDescriptorProto{
				Method: []*descriptor.MethodDescriptorProto{
					&descriptor.MethodDescriptorProto{},
				},
			},
			specs: []*httpproto.MethodSpec(nil),
		},
		{
			protoPackage: "test.package",
			serviceDesc: &descriptor.ServiceDescriptorProto{
				Method: []*descriptor.MethodDescriptorProto{
					&descriptor.MethodDescriptorProto{
						Options: &descriptor.MethodOptions{},
					},
				},
			},
			specs: []*httpproto.MethodSpec(nil),
		},
		{
			protoPackage: "test.package",
			serviceDesc: &descriptor.ServiceDescriptorProto{
				Name: proto.String("Users"),
				Method: []*descriptor.MethodDescriptorProto{
					&descriptor.MethodDescriptorProto{
						Name:       proto.String("Find"),
						InputType:  proto.String("test.package.FindRequest"),
						OutputType: proto.String("test.package.FindResponse"),
						Options: withExtension(t, &descriptor.MethodOptions{},
							&google_api.HttpRule{Pattern: &google_api.HttpRule_Get{Get: "/find"}}),
					},
				},
			},
			specs: []*httpproto.MethodSpec{
				&httpproto.MethodSpec{
					FullName:   "/test.package.Users/Find",
					InputType:  "test.package.FindRequest",
					OutputType: "test.package.FindResponse",
					Routes: []*httpproto.RouteSpec{
						&httpproto.RouteSpec{
							HTTPMethod:  "GET",
							PathPattern: "/find",
						},
					},
				},
			},
		},
	} {
		t.Run("", func(t *testing.T) {
			specs, err := FromServiceDescriptor(tt.protoPackage, tt.serviceDesc)
			if err != nil {
				if tt.specs != nil {
					t.Fatalf("FromServiceDescriptor; err = %v", err)
				}
				return
			}
			if have, want := specs, tt.specs; !reflect.DeepEqual(have, want) {
				t.Errorf("FromServiceDescriptor;\n%#v\n!=\n%#v", have, want)
			}
		})
	}
}

func TestRouteSpecs(t *testing.T) {
	for _, tt := range []struct {
		rules *google_api.HttpRule
		specs []*httpproto.RouteSpec
	}{
		{rules: nil, specs: nil},
		{
			rules: &google_api.HttpRule{Pattern: &google_api.HttpRule_Get{Get: "/read"}},
			specs: []*httpproto.RouteSpec{{HTTPMethod: "GET", PathPattern: "/read"}},
		},
		{
			rules: &google_api.HttpRule{
				Pattern: &google_api.HttpRule_Post{Post: "/update"}, Body: "*",
				AdditionalBindings: []*google_api.HttpRule{
					&google_api.HttpRule{
						Pattern: &google_api.HttpRule_Get{Get: "/wtf"}, Body: "body.with.get.is.somehow.ok",
						AdditionalBindings: []*google_api.HttpRule{
							&google_api.HttpRule{
								Pattern: &google_api.HttpRule_Patch{Patch: "/nested/additional_bindings/are/ignored"},
							},
						},
					},
				},
			},
			specs: []*httpproto.RouteSpec{
				{HTTPMethod: "POST", PathPattern: "/update", BodyField: "*"},
				{HTTPMethod: "GET", PathPattern: "/wtf", BodyField: "body.with.get.is.somehow.ok"},
			},
		},
	} {
		t.Run("", func(t *testing.T) {
			specs := routeSpecs(tt.rules)
			if have, want := specs, tt.specs; !reflect.DeepEqual(have, want) {
				t.Errorf("routeSpecs(%s);\n%#v\n!=\n%#v", tt.rules, have, want)
			}
		})
	}
}

func TestReadPattern(t *testing.T) {
	for _, tt := range []struct {
		rule *google_api.HttpRule
		kind string
		path string
	}{
		{rule: nil, kind: "", path: ""},
		{rule: &google_api.HttpRule{}, kind: "", path: ""},
		{rule: &google_api.HttpRule{Pattern: nil}, kind: "", path: ""},
		{rule: &google_api.HttpRule{Pattern: (*google_api.HttpRule_Get)(nil)}, kind: "", path: ""},
		{rule: &google_api.HttpRule{Pattern: &google_api.HttpRule_Get{}}, kind: "GET", path: ""},
		{rule: &google_api.HttpRule{Pattern: &google_api.HttpRule_Get{Get: "/pattern"}}, kind: "GET", path: "/pattern"},
		{rule: &google_api.HttpRule{Pattern: &google_api.HttpRule_Put{Put: "/pattern"}}, kind: "PUT", path: "/pattern"},
		{rule: &google_api.HttpRule{Pattern: &google_api.HttpRule_Post{Post: "/pattern"}}, kind: "POST", path: "/pattern"},
		{rule: &google_api.HttpRule{Pattern: &google_api.HttpRule_Delete{Delete: "/pattern"}}, kind: "DELETE", path: "/pattern"},
		{rule: &google_api.HttpRule{Pattern: &google_api.HttpRule_Patch{Patch: "/pattern"}}, kind: "PATCH", path: "/pattern"},

		{rule: &google_api.HttpRule{Pattern: (*google_api.HttpRule_Custom)(nil)}, kind: "", path: ""},
		{rule: &google_api.HttpRule{Pattern: &google_api.HttpRule_Custom{
			Custom: (*google_api.CustomHttpPattern)(nil)}}, kind: "", path: ""},
		{rule: &google_api.HttpRule{Pattern: &google_api.HttpRule_Custom{
			Custom: &google_api.CustomHttpPattern{Kind: "*", Path: "/pattern"}}}, kind: "*", path: "/pattern"},
		{rule: &google_api.HttpRule{Pattern: &google_api.HttpRule_Custom{
			Custom: &google_api.CustomHttpPattern{Kind: "HEAD", Path: "/pattern"}}}, kind: "HEAD", path: "/pattern"},
	} {
		t.Run("", func(t *testing.T) {
			kind, path := readPattern(tt.rule)
			if kind != tt.kind || path != tt.path {
				t.Errorf("readPattern(%s); %q, %q != %q, %q", tt.rule, kind, path, tt.kind, tt.path)
			}
		})
	}
}

func checkError(t *testing.T, logPrefix string, err error, errString string) {
	if err != nil {
		e := err.Error()
		if errString == "" {
			t.Errorf("%serr = %v", logPrefix, e)
			return
		}
		if !strings.Contains(e, errString) {
			t.Errorf("%s%q != %q", logPrefix, e, errString)
			return
		}
	} else if errString != "" {
		t.Errorf("%serr != %q", logPrefix, errString)
		return
	}
}

func setProtoFile(filename string, contents []byte) func() {
	prev := proto.FileDescriptor(filename)
	proto.RegisterFile(filename, contents)
	return func() { proto.RegisterFile(filename, prev) }
}

func TestGetProtoFileDescriptorBytes(t *testing.T) {
	testcase := func(filename string, errString string) func(t *testing.T) {
		return func(t *testing.T) {
			_, err := getProtoFileDescriptorBytes(filename)
			checkError(t, fmt.Sprintf("getProtoFileDescriptorBytes(%q); ", filename), err, errString)
		}
	}

	defer setProtoFile("xxx_test.missing.file", nil)()
	t.Run("missing file", testcase("xxx_test.missing.file", "not found"))

	defer setProtoFile("xxx_test.invalid.file", []byte("hello world"))()
	t.Run("not gzipped", testcase("xxx_test.invalid.file", "could not be decompressed"))

	var buf bytes.Buffer
	w := gzip.NewWriter(&buf)
	fmt.Fprintf(w, "hello world")
	w.Close()

	defer setProtoFile("xxx_test.full.file", buf.Bytes())()
	t.Run("gzipped", testcase("xxx_test.full.file", ""))

	defer setProtoFile("xxx_test.partial.file", buf.Bytes()[:buf.Len()-1])()
	t.Run("corrupt gzip", testcase("xxx_test.partial.file", "could not be decompressed"))

	t.Run("generated", testcase("code.justin.tv/common/alvin/internal/httpproto/loadroutes/loadtestproto/loadtest.proto", ""))
}

func TestGetProtoFileDescriptor(t *testing.T) {
	testcase := func(filename string, errString string) func(t *testing.T) {
		return func(t *testing.T) {
			_, err := getProtoFileDescriptor(filename)
			checkError(t, fmt.Sprintf("getProtoFileDescriptor(%q); ", filename), err, errString)
		}
	}

	defer setProtoFile("xxx_test.missing.file", nil)()
	t.Run("missing file", testcase("xxx_test.missing.file", "not found"))

	defer setProtoFile("xxx_test.invalid.file", []byte("hello world"))()
	t.Run("not gzipped", testcase("xxx_test.invalid.file", "could not be decompressed"))

	var buf bytes.Buffer
	w := gzip.NewWriter(&buf)
	fmt.Fprintf(w, "hello world")
	w.Close()

	defer setProtoFile("xxx_test.full.file", buf.Bytes())()
	t.Run("gzipped", testcase("xxx_test.full.file", "proto"))

	t.Run("generated", testcase("code.justin.tv/common/alvin/internal/httpproto/loadroutes/loadtestproto/loadtest.proto", ""))

	t.Run("confirm generated contents", func(t *testing.T) {
		fd, _ := getProtoFileDescriptor("code.justin.tv/common/alvin/internal/httpproto/loadroutes/loadtestproto/loadtest.proto")
		if have, want := fd.GetPackage(), "code.justin.tv.common.alvin.internal.httpproto.loadroutes.loadtestproto"; have != want {
			t.Errorf("getProtoFileDescriptor.GetPackage; %q != %q", have, want)
		}
	})
}

func TestGetMethodSpecs(t *testing.T) {
	t.Run("load generated method specs", func(t *testing.T) {
		fd, err := getProtoFileDescriptor("code.justin.tv/common/alvin/internal/httpproto/loadroutes/loadtestproto/loadtest.proto")
		if err != nil {
			t.Fatalf("getProtoFileDescriptor; err = %v", err)
		}

		// We need to add method annotations here, or the call to
		// getMethodSpecs will rightly return nothing. We're unable to place
		// the annotations in the .proto file because that would create a
		// dependency from this package's test files on the generated protobuf
		// code for the google.api.http annotation. That would in turn cause
		// the extension to be registered on init, so the tests for
		// proto_extension.go -- designed to function correctly either with or
		// without the extension registered -- would no longer be able to
		// reach 100% line coverage.
		//
		// This isn't a question of correctness -- we're fine on that front.
		// The problem is that this code is in the same package as
		// proto_extension.go, and in order to reach 100% test coverage the
		// tests for that code can't transitively depend on google.api.http.
		md := fd.Service[0].Method[0]
		if md.Options == nil {
			md.Options = &descriptor.MethodOptions{}
		}
		md.Options = withExtension(t, md.Options,
			&google_api.HttpRule{Pattern: &google_api.HttpRule_Get{Get: "/method"}})
		// And now, back to the test. That wasn't so bad, right?

		ms, err := getMethodSpecs(func() (*descriptor.FileDescriptorProto, error) { return fd, nil })
		checkError(t, "getMethodSpecs; ", err, "")
		if have, want := len(ms), 1; have < want {
			t.Errorf("getMethodSpecs.len; %d < %d", have, want)
		}
	})

	t.Run("no error on generated code", func(t *testing.T) {
		_, err := FromProtoFileName("code.justin.tv/common/alvin/internal/httpproto/loadroutes/loadtestproto/loadtest.proto")
		if err != nil {
			t.Fatalf("getProtoFileDescriptor; err = %v", err)
		}
	})

	t.Run("missing file", func(t *testing.T) {
		_, err := FromProtoFileName("code.justin.tv/common/alvin/internal/httpproto/loadroutes/loadtestproto/loadtest.prot")
		checkError(t, "getProtoFileDescriptor; ", err, "not found")
	})

	t.Run("invalid service descriptor", func(t *testing.T) {
		_, err := getMethodSpecs(func() (*descriptor.FileDescriptorProto, error) {
			return &descriptor.FileDescriptorProto{
				Package: proto.String("test.pkg"),
				Service: []*descriptor.ServiceDescriptorProto{nil},
			}, nil
		})
		checkError(t, "getMethodSpecs; ", err, "no service descriptor")
	})

	t.Run("return error", func(t *testing.T) {
		_, err := getMethodSpecs(func() (*descriptor.FileDescriptorProto, error) { return nil, fmt.Errorf("nope") })
		checkError(t, "getMethodSpecs; ", err, "nope")
	})
}
