package httproute

import (
	"net/url"
	"strings"
	"testing"
)

func TestEscapePathSegment(t *testing.T) {
	for i := rune(0); i <= 127; i++ {
		in := string([]rune{i})
		out := escapePathSegment(in)
		if strings.ContainsAny(out, "/;, ?#*") {
			t.Errorf("escapePathSegment(%q); %q contains unexpected character", in, out)
		}

		u, err := url.Parse("/" + out)
		if err != nil {
			t.Errorf("escapePathSegment(%q); %q is not a valid uri path: %v", in, out, err)
			continue
		}
		if have, want := strings.TrimPrefix(u.Path, "/"), in; have != want {
			t.Errorf("url.Parse(escapePathSegment(%q)).Path; %q != %q", in, have, want)
		}
	}
}

func TestExecuteTemplate(t *testing.T) {

	type entry struct {
		key string
		val string
	}

	lookupFunc := func(e []entry) func(string) (string, bool) {
		fn := func(key string) (val string, ok bool) {
			if len(e) == 0 || key != e[0].key {
				return "", false
			}
			val = e[0].val
			e = e[1:]
			return val, true
		}
		return fn
	}

	executeTemplate := func(tmpl string, lookup func(string) (string, bool)) (string, error) {
		template, err := ParseTemplate(tmpl)
		if err != nil {
			return "", err
		}
		err = template.Validate()
		if err != nil {
			return "", err
		}

		return template.Execute(lookup)
	}

	test := func(tmpl string, vals []entry, want string, errText string) func(t *testing.T) {
		fn := func(t *testing.T) {
			have, err := executeTemplate(tmpl, lookupFunc(vals))
			if err != nil {
				if errText == "" || !strings.Contains(err.Error(), errText) {
					t.Fatalf("tmpl.Execute(%q, %#v); err = %v", tmpl, vals, err)
				}
				return
			}
			if have != want {
				t.Errorf("tmpl.Execute(%q, %#v); %q != %q", tmpl, vals, have, want)
			}
		}
		return fn
	}

	t.Run("", test("/", nil, "", "invalid template"))
	t.Run("", test("/***", nil, "", "invalid template"))
	t.Run("", test("/{", nil, "", "invalid template"))

	t.Run("", test("/**", nil, "/", ""))

	t.Run("", test("/{foo}", nil, "", "lookup failed"))
	t.Run("", test("/{foo}", []entry{{key: "foo", val: ""}}, "", "does not match template"))
	t.Run("", test("/{foo}", []entry{{key: "foo", val: "hello"}}, "/hello", ""))
	t.Run("", test("/{foo}", []entry{{key: "foo", val: "hello/world"}}, "/hello%2Fworld", ""))
	t.Run("", test("/{foo=*}", []entry{{key: "foo", val: "hello/world"}}, "/hello%2Fworld", ""))
	t.Run("", test("/{foo=*/world}", []entry{{key: "foo", val: "hello/world"}}, "/hello/world", ""))
	t.Run("", test("/{foo=*/world}", []entry{{key: "foo", val: "hello"}}, "", "does not match template"))
	t.Run("", test("/{foo=*/world}", []entry{{key: "foo", val: "he/ll/o/world"}}, "", "does not match template"))

	t.Run("", test("/{foo}/world", []entry{{key: "foo", val: "hello"}}, "/hello/world", ""))
	t.Run("", test("/{foo}:world", []entry{{key: "foo", val: "hello"}}, "/hello:world", ""))

	t.Run("", test("/hello/**", nil, "/hello/", ""))

	t.Run("", test("/{foo}", []entry{{key: "bar", val: ""}}, "", "lookup failed"))

	t.Run("", test("/{foo=*}", []entry{{key: "foo", val: "hello"}}, "/hello", ""))
	t.Run("", test("/*", []entry{{key: "foo", val: "hello"}}, "/hello", "no data"))

	t.Run("", test("/**/world", []entry{{key: "foo", val: "hello"}}, "", "expansion token"))
	t.Run("", func(t *testing.T) {
		t.Skip("Validate does not yet check for ** in variable patterns")
		test("/{foo=**}/world", []entry{{key: "foo", val: "hello"}}, "", "expansion token")(t)
	})

	t.Run("", test("/{report_id}/report/bin/{program.name=**}",
		[]entry{{key: "report_id", val: "2006-01-02"}, {key: "program.name", val: "code.justin.tv/release/trace"}},
		"/2006-01-02/report/bin/code.justin.tv/release/trace", ""))
}
