package httproute

import (
	"fmt"
	"net/url"
	"reflect"
	"strings"
	"testing"
)

func TestParams(t *testing.T) {
	t.Run("non-matching", func(t *testing.T) {
		for _, tt := range []struct {
			tmpl string
			path string
			want string
		}{
			{
				tmpl: "/foo/bar",
				path: "/foo",
				want: "invariant broken",
			},
			{
				tmpl: "/**",
				path: "/:foo",
				want: "invariant broken",
			},
		} {
			tmpl, err := ParseTemplate(tt.tmpl)
			if err != nil {
				t.Fatalf("parseTemplate(%q); err = %v", tt.tmpl, err)
			}
			if tmpl.isMatch(tt.path) {
				t.Fatalf("parseTemplate(%q).isMatch(%q); true (bad test case)", tt.tmpl, tt.path)
			}

			var have interface{}
			func() {
				defer func() { have = recover() }()
				tmpl.capturePathParams(tt.path)
			}()

			if have, want := fmt.Sprint(have), tt.want; !strings.Contains(have, want) {
				t.Errorf("parseTemplate(%q).capturePathParams(%q);\n%#v\ndoes not contain\n%#v", tt.tmpl, tt.path, have, want)
			}
		}
	})

	for _, tt := range []struct {
		tmpl string
		path string
		want url.Values
	}{
		{
			tmpl: "/latest",
			path: "/latest",
			want: url.Values{},
		},
		{
			tmpl: "/{report_id}/report",
			path: "/2016-12-20-19-55-06.749755/report",
			want: url.Values{
				"report_id": []string{"2016-12-20-19-55-06.749755"},
			},
		},
		{
			tmpl: "/{report_id}/report/bin/{program.name=**}",
			path: "/2016-12-20-19-55-06.749755/report/bin/code.justin.tv/video/usher/api/usher",
			want: url.Values{
				"report_id":    []string{"2016-12-20-19-55-06.749755"},
				"program.name": []string{"code.justin.tv/video/usher/api/usher"},
			},
		},
		{
			tmpl: "/*/report/tx",
			path: "/2016-12-20-19-55-06.749755/report/tx",
			want: url.Values{},
		},
		{
			tmpl: "/{report_id}/report/tx/{transaction_id}",
			path: "/2016-12-20-19-55-06.749755/report/tx/79a35068bec54ad7b888c5c1df1821ff",
			want: url.Values{
				"report_id":      []string{"2016-12-20-19-55-06.749755"},
				"transaction_id": []string{"79a35068bec54ad7b888c5c1df1821ff"},
			},
		},
		{
			tmpl: "/{report_id}/report/bin/{program.name=**}",
			path: "/2016-12-20-19-55-06.749755/report/bin/",
			want: url.Values{
				"report_id":    []string{"2016-12-20-19-55-06.749755"},
				"program.name": []string{""},
			},
		},
		{
			tmpl: "/{report_id}/report/bin/{program.name=**}",
			path: "/2016-12-20-19-55-06.749755/report/bin",
			want: url.Values{
				"report_id":    []string{"2016-12-20-19-55-06.749755"},
				"program.name": []string{""},
			},
		},

		{
			tmpl: "/v2/{table_name=projects/*/instances/*/tables/*}:mutateRow",
			path: "/v2/projects/foo/instances/bar/tables/baz:mutateRow",
			want: url.Values{
				"table_name": []string{"projects/foo/instances/bar/tables/baz"},
			},
		},
	} {
		t.Run("", func(t *testing.T) {
			tmpl, err := ParseTemplate(tt.tmpl)
			if err != nil {
				t.Fatalf("parseTemplate(%q); err = %v", tt.tmpl, err)
			}
			if !tmpl.isMatch(tt.path) {
				t.Fatalf("parseTemplate(%q).isMatch(%q); false (bad test case)", tt.tmpl, tt.path)
			}
			if have, want := tmpl.capturePathParams(tt.path), tt.want; !reflect.DeepEqual(have, want) {
				t.Errorf("parseTemplate(%q).capturePathParams(%q);\n%#v\n!=\n%#v", tt.tmpl, tt.path, have, want)
			}
		})
	}
}
