package httproute

import (
	"reflect"
	"testing"
)

func TestSplitSegment(t *testing.T) {
	for _, tt := range []struct {
		in, want1, want2 string
	}{
		{"{report_id}/report/bin/{program.name=**}", "{report_id}", "report/bin/{program.name=**}"},
		{"report/bin/{program.name=**}", "report", "bin/{program.name=**}"},
		{"bin/{program.name=**}", "bin", "{program.name=**}"},
		{"{program.name=**}", "{program.name=**}", ""},

		{"*/report/tx", "*", "report/tx"},
		{"report/tx", "report", "tx"},
		{"tx", "tx", ""},

		// splitSegment is a recognizer for the segments' input language
		{"a/b", "a", "b"},
		{"a//b", "", ""},
		{":/b", "", ""},
		{"a/:", "", ""},
		{"a/.", "a", "."},
		{"a/=", "a", "="},
		{"a/{", "", ""},
		{"a/_{", "", ""},
		{"a/}", "", ""},
		{"a//", "", ""},
		{"a/;", "", ""},
		{"a/,", "", ""},
		{"a/*", "a", "*"},
		{"a/**", "a", "**"},
		{"a/***", "", ""},
		{"a/{b.c}", "a", "{b.c}"},
		{"a/{b..c}", "", ""},
	} {
		t.Run("splitSegment", func(t *testing.T) {
			have1, have2 := splitSegment(tt.in)
			if have1 != tt.want1 || have2 != tt.want2 {
				t.Errorf("splitSegment(%q); %q, %q != %q, %q", tt.in, have1, have2, tt.want1, tt.want2)
			}
		})
	}
}

func TestReadSegment(t *testing.T) {
	for _, tt := range []struct {
		in, want1, want2 string
	}{
		{"{report_id}", "", "{report_id}"},
		{"report", "report", ""},
		{"bin", "bin", ""},
		{"{program.name=**}", "", "{program.name=**}"},

		{"*", "*", ""},
		{"report", "report", ""},
		{"tx", "tx", ""},

		// readSegment is a recognizer for the segment's input language
		{"/", "", ""},
		{".", ".", ""},
		{"***", "", ""},
	} {
		tt := tt
		t.Run("readSegment", func(t *testing.T) {
			have1, have2 := readSegment(tt.in)
			if have1 != tt.want1 || have2 != tt.want2 {
				t.Errorf("readSegment(%q); %q, %q != %q, %q", tt.in, have1, have2, tt.want1, tt.want2)
			}
		})
	}
}

func TestParseTemplate(t *testing.T) {
	// Several of the test patterns don't specify explicit "Segments" to be
	// matched by the capture variables. This means they'll end up implicitly
	// matching a single path segment. For the sake of having smaller test
	// cases they'll all share the following declaration, which has that
	// meaning.
	matchOne := tmplSegments{
		tmplSegment{literal: "*", implicit: true},
	}

	for _, tt := range []struct {
		in  string
		out *Template
	}{
		{in: "/{report_id}/report",
			out: &Template{
				segments: tmplSegments{
					{variable: &tmplVariable{fieldPath: []string{"report_id"}, segments: matchOne}},
					{literal: "report"},
				},
			},
		},
		{in: "/{report_id}/report/bin/{program.name=**}",
			out: &Template{
				segments: tmplSegments{
					{variable: &tmplVariable{fieldPath: []string{"report_id"}, segments: matchOne}},
					{literal: "report"},
					{literal: "bin"},
					{variable: &tmplVariable{fieldPath: []string{"program", "name"}, segments: tmplSegments{{literal: "**"}}}},
				},
			},
		},
		{in: "/{report_id}/report/tx/{transaction_id}",
			out: &Template{
				segments: tmplSegments{
					{variable: &tmplVariable{fieldPath: []string{"report_id"}, segments: matchOne}},
					{literal: "report"},
					{literal: "tx"},
					{variable: &tmplVariable{fieldPath: []string{"transaction_id"}, segments: matchOne}},
				},
			},
		},
		{in: "/*/report/tx",
			out: &Template{
				segments: tmplSegments{
					{literal: "*"},
					{literal: "report"},
					{literal: "tx"},
				},
			},
		},
		{in: "/latest",
			out: &Template{
				segments: tmplSegments{
					{literal: "latest"},
				},
			},
		},

		{in: "/api/1.0/devices",
			out: &Template{
				segments: tmplSegments{
					{literal: "api"},
					{literal: "1.0"},
					{literal: "devices"},
				},
			},
		},

		// https://github.com/googleapis/googleapis/blob/master/google/bigtable/v2/bigtable.proto
		{in: "/v2/{table_name=projects/*/instances/*/tables/*}:mutateRow",
			out: &Template{
				segments: tmplSegments{
					{literal: "v2"},
					{variable: &tmplVariable{
						fieldPath: []string{"table_name"},
						segments: tmplSegments{
							{literal: "projects"},
							{literal: "*"},
							{literal: "instances"},
							{literal: "*"},
							{literal: "tables"},
							{literal: "*"},
						},
					}},
				},
				verb: "mutateRow",
			},
		},
	} {
		tt := tt
		t.Run("parseTemplate", func(t *testing.T) {
			tmpl, err := ParseTemplate(tt.in)
			if tt.out == nil && err == nil {
				t.Fatalf("parseTemplate(%q); err = nil", tt.in)
			}
			if err != nil {
				t.Fatalf("parseTemplate(%q); err = %q", tt.in, err)
			}
			if have, want := tmpl, tt.out; !reflect.DeepEqual(have, want) {
				t.Errorf("parseTemplate(%q);\n%#v\n!=\n%#v", tt.in, have, want)
			}
			if have, want := tmpl.String(), tt.in; have != want {
				t.Errorf("parseTemplate(%q).String(); %q", tt.in, have)
			}
		})
	}

	for _, tt := range []string{
		"",
		"/",
		":",
		":o",
		"/:",
		"/:o",
		"/***",
		"/***/*",
		"/*/",
		"/o/",
		"//",
		"//o",
		"o",
		"/{o=}",
		"/{}",
		"/o{o}",
		"/{o}o",
		"/{o..o}",
		"/{.o}",
		"/{o.}",
		"/{o=/}",
		"/{o=/o}",
		"/{o=o/}",
		"/{o+o}",
		"/{o*o}",
		"/{o/o}",
		"/{_o}",
		"/{0o}",
	} {
		t.Run("parseTemplate invalid", func(t *testing.T) {
			_, err := ParseTemplate(tt)
			if err == nil {
				t.Fatalf("parseTemplate(%q); err = nil", tt)
			}
		})
	}
}

func BenchmarkParseTemplate(b *testing.B) {
	fn := func(in string) {
		b.Run(in, func(b *testing.B) {
			for i := 0; i < b.N; i++ {
				_, err := ParseTemplate(in)
				if err != nil {
					b.Fatalf("parseTemplate(%q); err = %v", in, err)
				}
			}
		})
	}

	fn("/{report_id}/report")
	fn("/{report_id}/report/bin/{program.name=**}")
	fn("/{report_id}/report/tx/{transaction_id}")
	fn("/*/report/tx")
	fn("/latest")
	fn("/v2/{table_name=projects/*/instances/*/tables/*}:mutateRow")
	fn("/**")
}

func TestPathSegments(t *testing.T) {
	tmpl, err := ParseTemplate("/a/*/{v1}/{v2=b/c/*/d/*}/**")
	if err != nil {
		t.Fatalf("parseTemplate; err = %q", err)
	}
	if have, want := tmpl.pathSegments(), []string{"a", "*", "*", "b", "c", "*", "d", "*", "**"}; !reflect.DeepEqual(have, want) {
		t.Errorf("tmpl.pathSegments; %q != %q", have, want)
	}
}

func TestTemplateValidate(t *testing.T) {
	t.Run("nesting", func(t *testing.T) {
		err := (&Template{
			segments: tmplSegments{
				{variable: &tmplVariable{
					fieldPath: []string{"report_id"},
					segments: tmplSegments{
						{variable: &tmplVariable{fieldPath: []string{"transaction_id"}}},
					},
				}},
			},
		}).Validate()
		if err == nil {
			t.Fatalf("Validate; err = nil")
		}
	})

	t.Run("end stars", func(t *testing.T) {
		tmpl, err := ParseTemplate("/a/*/{v1}/{v2=b/c/*/d/*}/**")
		if err != nil {
			t.Fatalf("parseTemplate; err = %q", err)
		}
		err = tmpl.Validate()
		if err != nil {
			t.Fatalf("Validate; err = %q", err)
		}
	})

	t.Run("mid stars", func(t *testing.T) {
		tmpl, err := ParseTemplate("/a/**/{v1}/{v2=b/c/*/d/*}/**")
		if err != nil {
			t.Fatalf("parseTemplate; err = %q", err)
		}
		err = tmpl.Validate()
		if err == nil {
			t.Fatalf("Validate; err = nil")
		}
	})
}
