package restclient

import (
	"fmt"
	"net/url"
	"reflect"
	"sort"
	"strings"
	"testing"

	"code.justin.tv/common/alvin/internal/testproto"
	"github.com/golang/protobuf/proto"
	"github.com/golang/protobuf/ptypes/any"
	"github.com/golang/protobuf/ptypes/duration"
	"github.com/golang/protobuf/ptypes/empty"
	structpb "github.com/golang/protobuf/ptypes/struct"
	"github.com/golang/protobuf/ptypes/timestamp"
	"github.com/golang/protobuf/ptypes/wrappers"
)

func checkError(t *testing.T, logPrefix string, err error, errString string) {
	if err != nil {
		e := err.Error()
		if errString == "" {
			t.Errorf("%serr = %v", logPrefix, e)
			return
		}
		if !strings.Contains(e, errString) {
			t.Errorf("%s%q != %q", logPrefix, e, errString)
			return
		}
	} else if errString != "" {
		t.Errorf("%snil err != %q", logPrefix, errString)
		return
	}
}

func abortIfFailed(t *testing.T) {
	if t.Failed() {
		t.FailNow()
	}
}

func TestWalkProtobufStruct(t *testing.T) {
	t.Run("nil message", func(t *testing.T) {
		err := walkProtobufStruct(nil, func(_ string, _ *proto.Properties, _ reflect.Value) error {
			t.Fatalf("walkFunc called unexpectedly")
			return nil
		})
		checkError(t, "walkProtobufStruct; ", err, "nil message")
	})

	t.Run("user error during walk", func(t *testing.T) {
		eject := fmt.Errorf("eject!")
		err := walkProtobufStruct(&testproto.RequestA3{
			Deeply: &testproto.RequestA3_SubOne{
				Nested: &testproto.RequestA3_SubTwo{
					Field:     66,
					EasterEgg: []string{"up", "up"},
				},
				Buried: []string{"hello", "world"},
			},
		}, func(path string, _ *proto.Properties, _ reflect.Value) error {
			if path == "deeply.nested.field" {
				return eject
			}
			return nil
		})
		checkError(t, "walkProtobufStruct; ", err, "eject")
	})

	for _, message := range []proto.Message{
		&testproto.RequestA3{
			Deeply: &testproto.RequestA3_SubOne{
				Nested: &testproto.RequestA3_SubTwo{
					Field:     66,
					EasterEgg: []string{"up", "up"},
				},
				Buried: []string{"hello", "world"},
			},
		},
		&testproto.RequestA2{
			Deeply: &testproto.RequestA2_SubOne{
				Nested: &testproto.RequestA2_SubTwo{
					Field:     proto.Int64(66),
					EasterEgg: []string{"up", "up"},
				},
				Buried: []string{"hello", "world"},
			},
		},
	} {
		t.Run("visit fields in order", func(t *testing.T) {
			fields := []string{
				"deeply",
				"deeply.nested",
				"deeply.nested.field",
				"deeply.nested.easter_egg",
				"deeply.nested.loop",
				"deeply.buried",
			}
			walkProtobufStruct(message, func(path string, _ *proto.Properties, _ reflect.Value) error {
				if len(fields) == 0 {
					t.Fatalf("no more fields, but visiting %q", path)
				}
				if have, want := path, fields[0]; have != want {
					t.Errorf("walkProtobufStruct.path; %q != %q", have, want)
					return nil
				}
				fields = fields[1:]
				return nil
			})
			if len(fields) > 0 {
				t.Fatalf("unvisited fields: %q", fields)
			}
		})
	}
}

func TestLookupField(t *testing.T) {
	t.Run("nil message", func(t *testing.T) {
		_, err := lookupField(nil, "key")
		checkError(t, "lookupField; ", err, "cannot look up fields on nil message")
	})
	t.Run("repeated field", func(t *testing.T) {
		_, err := lookupField(&testproto.BitOfEverything2{}, "repeated_int32")
		checkError(t, "lookupField; ", err, "not valid for lookup")
	})
	t.Run("leaf field", func(t *testing.T) {
		fv, err := lookupField(
			&testproto.BitOfEverything2{
				Int64Value: &wrappers.Int64Value{Value: 17},
			},
			"int64_value.value")
		if err != nil {
			t.Fatalf("lookupField; err = %v", err)
		}
		if have, want := fv.Interface(), (interface{})(int64(17)); have != want {
			t.Errorf("lookupField; %#v != %#v", have, want)
		}
	})
}

func TestLeafFieldValues(t *testing.T) {
	t.Run("cyclic datastructure", func(t *testing.T) {
		msg := &testproto.RequestA3{
			Deeply: &testproto.RequestA3_SubOne{
				Nested: &testproto.RequestA3_SubTwo{
					Field:     66,
					EasterEgg: []string{"up", "up"},
				},
				Buried: []string{"hello", "world"},
			},
		}
		msg.Deeply.Nested.Loop = msg.Deeply.Nested
		u, err := leafFieldValues(msg)
		if err != nil {
			t.Fatalf("leafFieldValues; err = %s", err)
		}
		if have, want := u, (url.Values{
			"deeply.buried":            []string{"hello", "world"},
			"deeply.nested.easter_egg": []string{"up", "up"},
			"deeply.nested.field":      []string{"66"},
		}); have.Encode() != want.Encode() {
			t.Errorf("leafFieldValues;\n%s\n!=\n%s", have.Encode(), want.Encode())
		}
	})

	t.Run("nil message", func(t *testing.T) {
		_, err := leafFieldValues(nil)
		checkError(t, "leafFieldValues; ", err, "nil message")
	})

	t.Run("nil slice is not serialized", func(t *testing.T) {
		msg := &testproto.RequestA3{
			Deeply: &testproto.RequestA3_SubOne{
				Buried: nil,
			},
		}
		u, err := leafFieldValues(msg)
		if err != nil {
			t.Fatalf("leafFieldValues; err = %s", err)
		}
		if have, want := u, (url.Values{}); have.Encode() != want.Encode() {
			t.Errorf("leafFieldValues;\n%s\n!=\n%s", have.Encode(), want.Encode())
		}
	})

	t.Run("unknown well-known type", func(t *testing.T) {
		_, err := leafFieldValues(&badHolder{XXX_Badness: []*badWellKnown{&badWellKnown{}}})
		checkError(t, "leafFieldValues; ", err, "unknown well-known type")
	})
}

func TestStringFormat(t *testing.T) {
	testcase := func(val reflect.Value, out string) func(t *testing.T) {
		return func(t *testing.T) {
			if have, want := stringFormat(val), out; have != want {
				t.Errorf("stringFormat; %q != %q", have, want)
			}
		}
	}
	t.Run("nil", testcase(reflect.ValueOf(nil), ""))

	t.Run("string", testcase(reflect.ValueOf("hello world"), "hello world"))
	t.Run("[]byte", testcase(reflect.ValueOf([]byte("hello world~")), "aGVsbG8gd29ybGR-"))

	t.Run("int", testcase(reflect.ValueOf(int(42)), "42"))
	t.Run("int64", testcase(reflect.ValueOf(int64(42)), "42"))
	t.Run("int32", testcase(reflect.ValueOf(int32(42)), "42"))
	t.Run("int16", testcase(reflect.ValueOf(int16(42)), "42"))
	t.Run("int8", testcase(reflect.ValueOf(int8(42)), "42"))
	t.Run("uint", testcase(reflect.ValueOf(uint(42)), "42"))
	t.Run("uint64", testcase(reflect.ValueOf(uint64(42)), "42"))
	t.Run("uint32", testcase(reflect.ValueOf(uint32(42)), "42"))
	t.Run("uint16", testcase(reflect.ValueOf(uint16(42)), "42"))
	t.Run("uint8", testcase(reflect.ValueOf(uint8(42)), "42"))

	t.Run("float64", testcase(reflect.ValueOf(float64(1.21)), "1.21"))
	t.Run("float32", testcase(reflect.ValueOf(float32(1.21)), "1.21"))

	t.Run("bool", testcase(reflect.ValueOf(true), "true"))
	t.Run("bool", testcase(reflect.ValueOf(false), "false"))

	t.Run("Value", testcase(reflect.ValueOf(&structpb.Value{}), ""))
}

func TestBitOfEverything(t *testing.T) {
	testcase := func(msg proto.Message, vals url.Values) func(t *testing.T) {
		return func(t *testing.T) {
			u, err := leafFieldValues(msg)
			if err != nil {
				t.Fatalf("leafFieldValues; err = %v", err)
			}
			if have, want := u, vals; have.Encode() != want.Encode() {
				t.Errorf("leafFieldValues;\n%s\n!=\n%s", have.Encode(), want.Encode())

				allKeys := make(map[string]struct{})
				var keys []string
				for _, q := range []url.Values{u, vals} {
					for k := range q {
						if _, ok := allKeys[k]; !ok {
							keys = append(keys, k)
						}
						allKeys[k] = struct{}{}
					}
				}
				sort.Strings(keys)

				t.Logf("details:")
				for _, k := range keys {
					if have, want := (url.Values{k: u[k]}), (url.Values{k: vals[k]}); have.Encode() != want.Encode() {
						t.Logf("%q != %q", have.Encode(), want.Encode())
					}
				}
			}
		}
	}

	errcase := func(msg proto.Message, errString string) func(t *testing.T) {
		return func(t *testing.T) {
			_, err := leafFieldValues(msg)
			checkError(t, "leafFieldValues; ", err, errString)
		}
	}

	// Some test names include numbers at the beginning, indicating the field
	// numbers on the BitOfEverything message that are exercised in the test
	// case.

	t.Run("nil", testcase((*testproto.BitOfEverything2)(nil), url.Values{}))
	t.Run("nothing", testcase(&testproto.BitOfEverything2{}, url.Values{}))

	// basic repeated fields

	t.Run("1 repeated message", errcase(&testproto.BitOfEverything2{
		RepeatedMessage: []*testproto.BitOfEverything2_SmallMessage{
			{Value: proto.Int32(5)}, {Value: proto.Int32(7)},
		},
	}, "not valid leaf field"))

	t.Run("2 repeated int32", testcase(&testproto.BitOfEverything2{
		RepeatedInt32: []int32{5, 7},
	}, url.Values{"repeated_int32": []string{"5", "7"}}))

	t.Run("3 repeated bytes", testcase(&testproto.BitOfEverything2{
		RepeatedBytes: [][]byte{[]byte("hello"), []byte("world")},
	}, url.Values{"repeated_bytes": []string{"aGVsbG8", "d29ybGQ"}}))

	// maps

	t.Run("4 map of any", errcase(&testproto.BitOfEverything2{
		SingleAnyMap: map[bool]*any.Any{
			true:  &any.Any{TypeUrl: "https://localhost/path.to.Message", Value: []byte("hello world")},
			false: nil},
	}, "may not be repeated"))

	t.Run("5 map of string", errcase(&testproto.BitOfEverything2{
		SingleStringMap: map[string]string{"hey": "ho", "lets": "go"},
	}, "map fields are not supported"))

	// well-known types: special messages

	t.Run("11 any", errcase(&testproto.BitOfEverything2{
		Any: &any.Any{TypeUrl: "https://localhost/path.to.Message", Value: []byte("hello world")},
	}, "cannot format"))

	t.Run("12 duration", testcase(&testproto.BitOfEverything2{
		Duration: &duration.Duration{Seconds: 3, Nanos: 500000000},
	}, url.Values{"duration": []string{"3.500s"}}))

	t.Run("13 empty", testcase(&testproto.BitOfEverything2{
		Empty: &empty.Empty{},
	}, url.Values{"empty": []string{}}))

	t.Run("14 struct", errcase(&testproto.BitOfEverything2{
		Struct: &structpb.Struct{Fields: map[string]*structpb.Value{
			"nil":      nil,
			"blank":    &structpb.Value{},
			"nil null": &structpb.Value{Kind: (*structpb.Value_NullValue)(nil)},
			"null":     &structpb.Value{Kind: &structpb.Value_NullValue{}},
			"bool":     &structpb.Value{Kind: &structpb.Value_BoolValue{BoolValue: true}},
		}},
	}, "cannot format"))

	t.Run("15 timestamp", testcase(&testproto.BitOfEverything2{
		Timestamp: &timestamp.Timestamp{Seconds: 17, Nanos: 43000000},
	}, url.Values{"timestamp": []string{"1970-01-01T00:00:17.043Z"}}))

	// well-known types: wrappers

	t.Run("21 value nil null", errcase(&testproto.BitOfEverything2{
		Value: &structpb.Value{Kind: (*structpb.Value_NullValue)(nil)},
	}, "cannot format"))
	t.Run("21 value null", errcase(&testproto.BitOfEverything2{
		Value: &structpb.Value{Kind: &structpb.Value_NullValue{}},
	}, "cannot format"))
	t.Run("21 value bool", errcase(&testproto.BitOfEverything2{
		Value: &structpb.Value{Kind: &structpb.Value_BoolValue{BoolValue: true}},
	}, "cannot format"))

	t.Run("22 list value empty", errcase(&testproto.BitOfEverything2{
		ListValue: &structpb.ListValue{
			Values: []*structpb.Value{},
		},
	}, "cannot format"))
	t.Run("22 list value", errcase(&testproto.BitOfEverything2{
		ListValue: &structpb.ListValue{
			Values: []*structpb.Value{
				&structpb.Value{Kind: &structpb.Value_BoolValue{BoolValue: true}},
			},
		},
	}, "cannot format"))

	t.Run("23—31 values", testcase(&testproto.BitOfEverything2{
		DoubleValue: &wrappers.DoubleValue{Value: 1.21},
		FloatValue:  &wrappers.FloatValue{Value: 1.21},
		Int64Value:  &wrappers.Int64Value{Value: -5000000000},
		Uint64Value: &wrappers.UInt64Value{Value: 5000000000},
		Int32Value:  &wrappers.Int32Value{Value: 42},
		Uint32Value: &wrappers.UInt32Value{Value: 38},
		BoolValue:   &wrappers.BoolValue{Value: false},
		StringValue: &wrappers.StringValue{Value: "hello world"},
		BytesValue:  &wrappers.BytesValue{Value: []byte("hello world~")},
	}, url.Values{
		"value":        []string{},
		"list_value":   []string{},
		"double_value": []string{"1.21"},
		"float_value":  []string{"1.21"},
		"int64_value":  []string{"-5000000000"},
		"uint64_value": []string{"5000000000"},
		"int32_value":  []string{"42"},
		"uint32_value": []string{"38"},
		"bool_value":   []string{"false"},
		"string_value": []string{"hello world"},
		"bytes_value":  []string{"aGVsbG8gd29ybGR-"},
	}))

	// repeated wrappers

	t.Run("41 repeated int32 value", testcase(&testproto.BitOfEverything2{
		RepeatedInt32Value: []*wrappers.Int32Value{
			{Value: 3}, {Value: 2}, {Value: 1}, {Value: 0},
		},
	}, url.Values{"repeated_int32_value": []string{"3", "2", "1", "0"}}))

	t.Run("42 repeated bytes value", testcase(&testproto.BitOfEverything2{
		RepeatedBytesValue: []*wrappers.BytesValue{
			{Value: []byte(nil)}, {Value: []byte(":)")}, {Value: []byte("")},
		},
	}, url.Values{"repeated_bytes_value": []string{"", "Oik", ""}}))

	// oneof

	t.Run("51 oneof string", testcase(&testproto.BitOfEverything2{
		One: &testproto.BitOfEverything2_OneString{
			OneString: "hi",
		},
	}, url.Values{"one_string": []string{"hi"}}))

	t.Run("52 oneof double value", testcase(&testproto.BitOfEverything2{
		One: &testproto.BitOfEverything2_OneDoubleValue{
			OneDoubleValue: &wrappers.DoubleValue{Value: 1.21},
		},
	}, url.Values{"one_double_value": []string{"1.21"}}))

	t.Run("53 oneof small message", testcase(&testproto.BitOfEverything2{
		One: &testproto.BitOfEverything2_OneMessage{
			OneMessage: &testproto.BitOfEverything2_SmallMessage{Value: proto.Int32(42)},
		},
	}, url.Values{"one_message.value": []string{"42"}})) // nested field, separated by dot

	t.Run("54 oneof bytes", testcase(&testproto.BitOfEverything2{
		One: &testproto.BitOfEverything2_OneBytes{
			OneBytes: []byte(":)"),
		},
	}, url.Values{"one_bytes": []string{"Oik"}}))

	// check unset fields in proto2/proto3
	t.Run("61 single message proto2", testcase(&testproto.BitOfEverything2{
		SingleMessage: &testproto.BitOfEverything2_SmallMessage{Value: proto.Int32(22)},
	}, url.Values{"single_message.value": []string{"22"}}))
	t.Run("61 single message proto3", testcase(&testproto.BitOfEverything3{
		SingleMessage: &testproto.BitOfEverything3_SmallMessage{Value: 22},
	}, url.Values{"single_message.value": []string{"22"}}))
}

func TestWellKnownStringFormat(t *testing.T) {
	testcase := func(pb proto.Message, str string, ok bool) func(t *testing.T) {
		return func(t *testing.T) {
			haveStr, haveOk, err := wellKnownStringFormat(pb)
			if err != nil {
				t.Fatalf("wellKnownStringFormat(%v); err = %v", pb, err)
			}
			if haveStr != str || haveOk != ok {
				t.Errorf("wellKnownStringFormat(%v); %q, %t != %q, %t", pb, haveStr, haveOk, str, ok)
			}
		}
	}

	errcase := func(pb proto.Message, errString string) func(t *testing.T) {
		return func(t *testing.T) {
			_, _, err := wellKnownStringFormat(pb)
			checkError(t, fmt.Sprintf("wellKnownStringFormat(%v); ", pb),
				err, errString)
		}
	}

	t.Run("user type", testcase(&testproto.BitOfEverything2{}, "", false))
	t.Run("empty", testcase(&empty.Empty{}, "", false))
	t.Run("duration", testcase(&duration.Duration{Seconds: 3, Nanos: 500000000},
		"3.500s", true))
	t.Run("timestamp", testcase(&timestamp.Timestamp{Seconds: 17, Nanos: 43000000},
		"1970-01-01T00:00:17.043Z", true))
	t.Run("double", testcase(&wrappers.DoubleValue{Value: 1.21}, "1.21", true))
	t.Run("float", testcase(&wrappers.FloatValue{Value: 1.21}, "1.21", true))
	t.Run("int64", testcase(&wrappers.Int64Value{Value: -5000000000}, "-5000000000", true))
	t.Run("uint64", testcase(&wrappers.UInt64Value{Value: 5000000000}, "5000000000", true))
	t.Run("int32", testcase(&wrappers.Int32Value{Value: -42}, "-42", true))
	t.Run("uint32", testcase(&wrappers.UInt32Value{Value: 17}, "17", true))
	t.Run("string", testcase(&wrappers.StringValue{Value: "hi there"}, "hi there", true))
	t.Run("bytes", testcase(&wrappers.BytesValue{Value: []byte(":)")}, "Oik", true))
	t.Run("bool", testcase(&wrappers.BoolValue{Value: true}, "true", true))

	t.Run("bad well known", errcase(&badWellKnown{}, "unknown"))

	t.Run("bad json", errcase(&badJSONHolder{}, "error calling MarshalJSON"))
}

// badHolder is a test protobuf message containing an unknown "well-known"
// type.
type badHolder struct {
	XXX_Badness []*badWellKnown `protobuf:"bytes,1,rep,name=xxx_badness"`
}

func (_ *badHolder) ProtoMessage()  {}
func (_ *badHolder) Reset()         {}
func (_ *badHolder) String() string { return "bad holder" }

// badWellKnown is a "well-known" protobuf type that is unknown to our code.
type badWellKnown struct {
}

func (_ *badWellKnown) ProtoMessage()             {}
func (_ *badWellKnown) Reset()                    {}
func (_ *badWellKnown) String() string            { return "bad" }
func (_ *badWellKnown) XXX_WellKnownType() string { return "bad" }

// badJSONHolder is a test protobuf message containing a field which fails to
// be represented as a string.
type badJSONHolder struct {
	XXX_BadJSON badJSON
}

func (_ *badJSONHolder) ProtoMessage()             {}
func (_ *badJSONHolder) Reset()                    {}
func (_ *badJSONHolder) String() string            { return "bad json holder" }
func (_ *badJSONHolder) XXX_WellKnownType() string { return "BytesValue" }

// badJSON fails to serialize itself as JSON, to be used as a field in a test
// protobuf message.
type badJSON int

func (_ badJSON) MarshalJSON() ([]byte, error) { return nil, fmt.Errorf("bad json") }

func TestEachRepeat(t *testing.T) {
	t.Run("", func(t *testing.T) {
		err := eachRepeat(reflect.ValueOf(0), func(key, val string) {
			t.Fatalf("eachRepeat called closure with %q, %q", key, val)
		})
		checkError(t, "eachRepeat; ", err, "unhandled kind")
	})
}
