package combiner

import "testing"

func TestRecordCombinerAdd(t *testing.T) {
	// Set flush timeout to a very high value so we don't have any
	// time-based flushes.
	flushTime := assertionTimeout * 10000

	rc := NewRecordCombiner(1, 1000, flushTime)

	// Adding a record should not error
	rc.Add(fakeRecord(0))
	assertNoError(t, rc.Errors())

	// After waiting briefly, the RC should still not have flushed.
	assertEmpty(t, rc.Blobs())

	// After closing, the RC should flush a blob.
	rc.Close()
	assertAvailable(t, rc.Blobs())
}

func TestRecordCombinerConsume(t *testing.T) {
	records := make(chan Record, 64)

	// Set flush timeout to a very high value so we don't have any
	// time-based flushes.
	flushTime := assertionTimeout * 10000

	rc := NewRecordCombiner(1, 1000, flushTime)
	rc.Consume(records)

	// Initially, nothing should be coming out of the combiner
	assertEmpty(t, rc.Blobs())
	assertNoError(t, rc.Errors())

	// Add a record. Nothing should happen because of our long flush time.
	records <- fakeRecord(0)
	assertEmpty(t, rc.Blobs())
	assertNoError(t, rc.Errors())

	// Closing the input channel should close the RC and flush.
	close(records)
	assertAvailable(t, rc.Blobs())
	assertNoError(t, rc.Errors())
}

func TestShardID(t *testing.T) {
	rc := NewRecordCombiner(0, 0, 0)

	type testcase struct {
		idx, n int64
		want   string
	}

	testcases := []testcase{
		testcase{0, 3, "0"},
		testcase{1, 3, "113427455640312821154458202477256070485"},
		testcase{2, 3, "226854911280625642308916404954512140970"},
	}

	for _, tc := range testcases {
		have := rc.shardID(tc.idx, tc.n)
		if have != tc.want {
			t.Errorf("shardID(%d, %d) wrong have=%s want=%s", tc.idx, tc.n, have, tc.want)
		}
	}
}
