package combiner

import (
	"testing"
	"time"
)

func TestShardSize(t *testing.T) {
	var b *Blob
	id := "1"
	flushTime := assertionTimeout * 10000
	ch := make(chan *Blob, 64)
	errs := make(chan error, 64)
	maxSize := 1000
	shard := newShard(id, maxSize, flushTime, ch, errs)

	// Test a record being too big
	shard.add(fakeRecord(maxSize + 1))
	assertError(t, errs)
	assertEmpty(t, ch)

	// Test a well-sized record, then one that's too big
	shard.add(fakeRecord(maxSize - 1))
	assertNoError(t, errs)
	assertEmpty(t, ch)

	shard.add(fakeRecord(maxSize + 1))
	assertError(t, errs)

	b = assertAvailable(t, ch)
	if b.Key != id {
		t.Fatal("Incorrect blob id")
	}
	if have, want := len(b.Data), maxSize-1; have != want {
		t.Fatalf("Incorrect blob size. have=%d want=%d", have, want)
	}

	// Test hitting maxSize exactly
	shard.add(fakeRecord(maxSize))
	assertNoError(t, errs)

	b = assertAvailable(t, ch)
	if b.Key != id {
		t.Fatal("Incorrect blob id")
	}
	if have, want := len(b.Data), maxSize; have != want {
		t.Fatalf("Incorrect blob size. have=%d want=%d", have, want)
	}

	// Test overflowing the maxSize
	shard.add(fakeRecord(maxSize - 1))
	assertNoError(t, errs)
	assertEmpty(t, ch)

	shard.add(fakeRecord(2))
	assertNoError(t, errs)
	b = assertAvailable(t, ch)
	if b.Key != id {
		t.Fatal("Incorrect blob id")
	}
	if have, want := len(b.Data), maxSize-1; have != want {
		t.Fatalf("Incorrect blob size. have=%d want=%d", have, want)
	}

	shard.close()
}

func TestShardTime(t *testing.T) {
	flushTime := assertionTimeout * 5
	ch := make(chan *Blob, 64)
	errs := make(chan error, 64)
	shard := newShard("", 1000, flushTime, ch, errs)

	// No flushes if we don't add records
	time.Sleep(flushTime * 2)
	assertEmpty(t, ch)

	// Flush if we add any record and wait past maxTime
	// even if Encode() returns len()==0
	shard.add(fakeRecord(0))
	assertNoError(t, errs)
	assertEmpty(t, ch)
	time.Sleep(flushTime)
	assertAvailable(t, ch)

	// Don't flush if we flush due to exceeding maxSize
	shard.add(fakeRecord(1000))
	assertNoError(t, errs)
	assertAvailable(t, ch)

	time.Sleep(flushTime)
	assertEmpty(t, ch)

	shard.close()
}

func TestShardClose(t *testing.T) {
	// No flushes if we don't add records
	flushTime := assertionTimeout * 10000
	ch := make(chan *Blob, 64)
	errs := make(chan error, 64)
	shard := newShard("", 1000, flushTime, ch, errs)
	shard.close()
	assertEmpty(t, ch)
	assertNoError(t, errs)

	// Flush if we add any record and close
	// even if Encode() returns len()==0
	ch = make(chan *Blob, 64)
	errs = make(chan error, 64)
	shard = newShard("", 1000, flushTime, ch, errs)
	shard.add(fakeRecord(0))
	assertNoError(t, errs)
	shard.close()
	assertAvailable(t, ch)
	assertNoError(t, errs)
}
