package autoprof

import (
	"bytes"
	"context"
	"encoding/json"
	"expvar"
	"testing"
)

func readAll(source *DataSource) ([]byte, error) {
	var buf bytes.Buffer
	err := source.WriteTo(context.Background(), &buf)
	if err != nil {
		return nil, err
	}
	return buf.Bytes(), nil
}

func TestExpvarWriter(t *testing.T) {
	t.Run("valid-json", func(t *testing.T) {
		buf, err := readAll(expvarSource())
		if err != nil {
			t.Fatalf("readAll; err = %v", err)
		}
		v := make(map[string]interface{})
		err = json.Unmarshal(buf, &v)
		if err != nil {
			t.Fatalf("json.Unmarshal; err = %v", err)
		}
		_, ok := v["cmdline"]
		if !ok {
			t.Errorf("cmdline value not present")
		}
	})

	t.Run("name", func(t *testing.T) {
		buf, err := readAll(expvarStyleSource(func(f func(expvar.KeyValue)) {
			v := expvar.NewString("")
			v.Set("world")
			f(expvar.KeyValue{
				Key:   "hello",
				Value: v,
			})
		}))
		if err != nil {
			t.Fatalf("readAll; err = %v", err)
		}
		if have, want := string(buf), `
{
"hello": "world"
}
`[1:]; have != want {
			t.Errorf("output:\n%q\n!=\n%q", have, want)
		}
	})
}
