package tdigest

import (
	"bytes"
	"testing"

	"github.com/davecgh/go-spew/spew"
)

func TestFuzzPanicRegressions(t *testing.T) {
	// This test contains a list of byte sequences discovered by
	// github.com/dvyukov/go-fuzz which, at one time, caused tdigest to panic. The
	// test just makes sure that they no longer cause a panic.
	testcase := func(crasher []byte) func(*testing.T) {
		return func(t *testing.T) {
			v := new(TDigest)
			err := v.UnmarshalBinary(crasher)
			if err != nil {
				return
			}
			remarshaled, err := v.MarshalBinary()
			if err != nil {
				t.Fatalf("marshal error: %v", err)
			}

			if !bytes.HasPrefix(crasher, remarshaled) {
				t.Fatalf("not equal: \n%v\nvs\n%v", crasher, remarshaled)
			}

			for q := float64(0.1); q <= 1.0; q += 0.05 {
				prev, this := v.Quantile(q-0.1), v.Quantile(q)
				if prev-this > 1e-100 { // Floating point math makes this slightly imprecise.
					t.Logf("v: %s", spew.Sprint(v))
					t.Logf("q: %v", q)
					t.Logf("prev: %v", prev)
					t.Logf("this: %v", this)
					t.Fatal("quantiles should only increase")
				}
			}

			v.Add(1, 1)
		}
	}
	t.Run("fuzz1", testcase([]byte{
		0x01, 0x00, 0x00, 0x00, 0x30, 0x30, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0xfc,
	}))
	t.Run("fuzz2", testcase([]byte{
		0x01, 0x00, 0x00, 0x00, 0xdb, 0x46, 0x5f, 0xbd,
		0xdb, 0x46, 0x00, 0xbd, 0xe0, 0xdf, 0xca, 0xab,
		0x37, 0x31, 0x37, 0x32, 0x37, 0x33, 0x37, 0x34,
		0x37, 0x35, 0x37, 0x36, 0x37, 0x37, 0x37, 0x38,
		0x37, 0x39, 0x28,
	}))
	t.Run("fuzz3", testcase([]byte{
		0x80, 0x0c, 0x01, 0x00, 0x00, 0x00, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x02, 0x00,
		0x00, 0x00, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x30, 0xbf,
	}))
	t.Run("fuzz4", testcase([]byte{
		0x80, 0x0c, 0x01, 0x00, 0x00, 0x00, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x02, 0x00,
		0x00, 0x00, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x30, 0x63, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x30, 0x4e,
	}))
	t.Run("fuzz5", testcase([]byte{
		0x80, 0x0c, 0x01, 0x00, 0x00, 0x00, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x02, 0x00,
		0x00, 0x00, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x30, 0x00, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x30, 0x00, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
		0x92, 0x00,
	}))
}
