package twitchconsulapi

import (
	"errors"
	"fmt"
	"log"
	"net/http"
	"regexp"
	"sync"

	"github.com/hashicorp/consul/api"
)

const (
	MaxConcurrency = 20
)

var (
	fqdnRE *regexp.Regexp
)

func init() {
	fqdnRE = regexp.MustCompile("fqdn=(.*)")
}

type DCtoNodes struct {
	DC    string   `json:"dc"`
	Nodes []string `json:"nodes"`
}

func NewClient(consulHostname string, client *http.Client) (*api.Client, error) {
	if consulHostname == "" {
		consulHostname = "localhost:8500"
	}
	if client == nil {
		client = http.DefaultClient
	}
	return api.NewClient(&api.Config{
		Address:    consulHostname,
		HttpClient: client,
	})
}

// GetAliveHostnames will search through all datacenters and return a list of
// hostnames that are running a service with a matching tag. If the tag is an
// empty string it will return all fqdns with the service.
func GetAliveHostnames(client *api.Client, services []string, tag string, skipUndeployable bool) ([]string, error) {
	var hostnames []string

	dcs, err := setupDatacenters(client)
	if err != nil {
		return nil, err
	}

	for _, service := range services {
		newHosts, lserr := mapLookupService(dcs, service, tag, skipUndeployable)
		if lserr != nil {
			return nil, lserr
		}
		hostnames = append(hostnames, newHosts...)
	}

	return unique(hostnames), nil
}

// GetAliveHostnamesWithDCs will search through all datacenters and return a map of
// datacenter to hostnames that are running a service with a matching tag. If the tag is an
// empty string it will return all fqdns with the service.
func GetAliveHostnamesWithDCs(client *api.Client, services []string, tag string, skipUndeployable bool) ([]DCtoNodes, error) {
	DCtoHostnames := make(map[string][]string)
	DCtoNodesList := []DCtoNodes{}
	dcs, err := setupDatacenters(client)
	if err != nil {
		return nil, err
	}

	for _, service := range services {
		serviceDCstoHostnames, lserr := mapLookupServiceWithDCs(dcs, service, tag, skipUndeployable)
		if lserr != nil {
			return nil, lserr
		}
		for dcName, hosts := range serviceDCstoHostnames {
			if existingHosts, ok := DCtoHostnames[dcName]; ok {
				//Unique is called because multiple services can have the same hostname
				DCtoHostnames[dcName] = unique(append(existingHosts, hosts...))
			} else {
				DCtoHostnames[dcName] = hosts
			}
		}
	}

	for dc, nodes := range DCtoHostnames {
		DCtoNodesList = append(DCtoNodesList, DCtoNodes{DC: dc, Nodes: nodes})
	}

	return DCtoNodesList, nil
}

func unique(list []string) []string {
	set := map[string]bool{}

	for _, item := range list {
		set[item] = true
	}

	res := []string{}
	for item, _ := range set {
		res = append(res, item)
	}

	return res
}

// parseFQDN will look over a list of tags and find the one marking the FQDN
// and return that.
func parseFQDN(tags []string) (string, error) {
	for _, tag := range tags {
		matches := fqdnRE.FindSubmatch([]byte(tag))
		if len(matches) > 0 {
			return string(matches[1]), nil
		}
	}

	return "", errors.New("could not find fqdn tag")
}

// setupDatacenters will lookup all of the datacenters and create a new
// Datacenter object per datacenter.
func setupDatacenters(client *api.Client) (map[string]*Datacenter, error) {
	var wait sync.WaitGroup
	dcs := map[string]*Datacenter{}

	datacenterNames, err := client.Catalog().Datacenters()
	if err != nil {
		return nil, fmt.Errorf("error getting datacenters: %v", err)
	}

	for _, dcName := range datacenterNames {
		var err error
		var dc *Datacenter
		dc, err = NewDatacenter(dcName, client)
		if err != nil {
			// Skip broken datacenters.
			log.Printf("Error setting up datacenter %q: %v", dcName, err)
			continue
		}

		wait.Add(1)
		go func(localDC *Datacenter) {
			defer wait.Done()
			if err := localDC.GetData(); err != nil {
				log.Printf("Error fetching data for datacenter %q: %v", localDC.Name, err)
				return
			}
		}(dc)

		dcs[dc.Name] = dc
	}
	wait.Wait()

	return dcs, nil
}

// mapLookupService will parallel map LookupService in each datacenter.
func mapLookupService(dcs map[string]*Datacenter, service, tag string, skipUndeployable bool) ([]string, error) {
	var waitGroup sync.WaitGroup
	hostnames := []string{}
	hostChan := make(chan []string, len(dcs))
	errChan := make(chan error)

	// Spawn all of the functions to lookup the hosts:
	waitGroup.Add(len(dcs))
	for _, dc := range dcs {
		go func(localDC *Datacenter) {
			defer waitGroup.Done()

			hosts, err := localDC.LookupService(service, tag, skipUndeployable)
			if err != nil {
				log.Printf("Error looking up service %q in datacenter %q: %v", service, localDC.Name, err)
				errChan <- err
				return
			}
			hostChan <- hosts
		}(dc)
	}

	// Spawn a function to close the channel:
	go func() {
		waitGroup.Wait()
		close(hostChan)
		close(errChan)
	}()

	if err := <-errChan; err != nil {
		return hostnames, err
	}

	// Merge the hosts:
	for newHosts := range hostChan {
		hostnames = append(hostnames, newHosts...)
	}

	return hostnames, nil
}

func mapLookupServiceWithDCs(dcs map[string]*Datacenter, service, tag string, skipUndeployable bool) (map[string][]string, error) {
	var waitGroup sync.WaitGroup
	type ChanResult struct {
		DCName string
		Hosts  []string
	}
	DCtoHostnames := make(map[string][]string)
	hostChan := make(chan ChanResult, len(dcs))
	errChan := make(chan error)

	// Spawn all of the functions to lookup the hosts:
	waitGroup.Add(len(dcs))
	for _, dc := range dcs {
		go func(localDC *Datacenter) {
			defer waitGroup.Done()

			hosts, err := localDC.LookupService(service, tag, skipUndeployable)
			if err != nil {
				log.Printf("Error looking up service %q in datacenter %q: %v", service, localDC.Name, err)
				errChan <- err
				return
			}
			hostChan <- ChanResult{localDC.Name, hosts}
		}(dc)
	}

	// Spawn a function to close the channel:
	go func() {
		waitGroup.Wait()
		close(hostChan)
		close(errChan)
	}()

	if err := <-errChan; err != nil {
		return DCtoHostnames, err
	}

	// Merge the hosts:
	for returnedResults := range hostChan {
		DCtoHostnames[returnedResults.DCName] = returnedResults.Hosts
	}

	return DCtoHostnames, nil
}
