package twitchconsulapi

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"reflect" //for map equivalence checking
	"testing"
)

// Used to generate fake serviceInfo
type serviceInfoDef struct {
	Address        string   `json:"Address"`
	Node           string   `json:"Node"`
	Dc             string   `json:"-"`
	ServiceId      string   `json:"ServiceID"`
	ServicePort    int      `json:"ServicePort"`
	ServiceAddress string   `json:"ServiceAddress"`
	ServiceTags    []string `json:"ServiceTags"`
	ServiceName    string   `json:"ServiceName"`
}

func (s *serviceInfoDef) init() {
	s.ServicePort = 0
	s.ServiceAddress = ""
	s.ServiceTags = []string{fmt.Sprintf("fqdn=%s.%s", s.Node, s.Dc), "lsbdistcodename=precise"}
	s.ServiceName = s.ServiceId
}

func fakeConsul() *httptest.Server {
	mux := http.NewServeMux()
	mux.HandleFunc("/v1/catalog/datacenters", fakeDatacenters)
	mux.HandleFunc("/v1/catalog/service/nodeinfo", fakeNodeInfo)
	mux.HandleFunc("/v1/catalog/service/nodeinfo2", fakeNodeInfo2)
	mux.HandleFunc("/v1/catalog/service/nodeinfo3", fakeNodeInfo3)
	mux.HandleFunc("/v1/health/state/passing", fakeStatusInfo)

	ts := httptest.NewServer(mux)
	return ts
}

func fakeConsulDCs() *httptest.Server {
	mux := http.NewServeMux()
	mux.HandleFunc("/v1/catalog/datacenters", fakeDatacenters)
	mux.HandleFunc("/v1/catalog/service/nodeinfo", fakeNodeInfo)
	mux.HandleFunc("/v1/catalog/service/nodeinfo2", fakeNodeInfo2)
	mux.HandleFunc("/v1/catalog/service/nodeinfo3", fakeNodeInfo3)
	mux.HandleFunc("/v1/health/state/passing", fakeStatusInfoDCs)

	ts := httptest.NewServer(mux)
	return ts
}

func fakeDatacenters(w http.ResponseWriter, r *http.Request) {
	fmt.Fprintf(w, `["test1", "test2", "test3"]`)
}

func fakeStatusInfo(w http.ResponseWriter, r *http.Request) {
	dc := r.URL.Query().Get("dc")

	switch dc {
	case "test1":
		fmt.Fprintf(w, `[
		{"ServiceName": "Serf Health", "ServiceID": "serfHealth", "Output": "", "Notes": "",
		 "Status": "passing", "Name": "Serf Health", "CheckID": "serfHealth", "Node": "serf-1-app-0.dev"}
		]`)
	case "test2":
		fmt.Fprintf(w, "[]")
	case "test3":
		w.WriteHeader(http.StatusInternalServerError)
	}
}

func fakeStatusInfoDCs(w http.ResponseWriter, r *http.Request) {
	dc := r.URL.Query().Get("dc")

	switch dc {
	case "test1":
		fmt.Fprintf(w, `[
		{"ServiceName": "Serf Health", "ServiceID": "serfHealth", "Output": "", "Notes": "",
		 "Status": "passing", "Name": "Serf Health", "CheckID": "serfHealth", "Node": "serf-1-app-0.dev"}
		]`)
	case "test2":
		fmt.Fprintf(w, `[
		{"ServiceName": "Serf Health", "ServiceID": "serfHealth", "Output": "", "Notes": "",
		 "Status": "passing", "Name": "Serf Health", "CheckID": "serfHealth", "Node": "other-app-0.dev"}
		]`)
	case "test3":
		w.WriteHeader(http.StatusInternalServerError)
	}
}

func fakeNodeInfo(w http.ResponseWriter, r *http.Request) {
	dc := r.URL.Query().Get("dc")

	switch dc {
	case "test1":
		serf1 := serviceInfoDef{Node: "serf-1-app-0.dev", Address: "127.1.6.32", Dc: "test1.justin.tv", ServiceId: "nodeinfo"}
		serf1.init()
		fakeServices := []serviceInfoDef{serf1}
		b, _ := json.Marshal(fakeServices)
		fmt.Fprintf(w, string(b))
	case "test2":
		serf2 := serviceInfoDef{Node: "serf-2-app-0.dev", Address: "127.1.5.18", Dc: "test2.justin.tv", ServiceId: "nodeinfo2"}
		other := serviceInfoDef{Node: "other-app-0.dev", Address: "127.1.5.19", Dc: "test2.justin.tv", ServiceId: "nodeinfo3"}
		serf2.init()
		other.init()
		fakeServices := []serviceInfoDef{serf2, other}
		//NOTE: I'm marshalling because I want commas
		b, _ := json.Marshal(fakeServices)
		fmt.Fprintf(w, string(b))
	case "test3":
		w.WriteHeader(http.StatusInternalServerError)
	}
}

func fakeNodeInfo2(w http.ResponseWriter, r *http.Request) {
	dc := r.URL.Query().Get("dc")

	switch dc {
	case "test1":
		serf1 := serviceInfoDef{Node: "serf-1-app-0.dev", Address: "127.1.6.32", Dc: "test1.justin.tv", ServiceId: "nodeinfo2"}
		serf1.init()
		fakeServices := []serviceInfoDef{serf1}
		b, _ := json.Marshal(fakeServices)
		fmt.Fprintf(w, string(b))
	case "test2":
		serf2 := serviceInfoDef{Node: "serf-2-app-0.dev", Address: "127.1.5.18", Dc: "test2.justin.tv", ServiceId: "nodeinfo2"}
		other := serviceInfoDef{Node: "other-app-0.dev", Address: "127.1.5.19", Dc: "test2.justin.tv", ServiceId: "nodeinfo3"}
		serf2.init()
		other.init()
		fakeServices := []serviceInfoDef{serf2, other}
		b, _ := json.Marshal(fakeServices)
		fmt.Fprintf(w, string(b))
	case "test3":
		w.WriteHeader(http.StatusInternalServerError)
	}
}

func fakeNodeInfo3(w http.ResponseWriter, r *http.Request) {
	dc := r.URL.Query().Get("dc")

	switch dc {
	case "test1":
		fmt.Fprintf(w, "[]")
	case "test2":
		other := serviceInfoDef{Node: "other-app-0.dev", Address: "127.1.5.19", Dc: "test2.justin.tv", ServiceId: "nodeinfo3"}
		other.init()
		fakeServices := []serviceInfoDef{other}
		b, _ := json.Marshal(fakeServices)
		fmt.Fprintf(w, string(b))
	case "test3":
		w.WriteHeader(http.StatusInternalServerError)
	}
}

func TestParseFQDN(t *testing.T) {
	tags := []string{"fqdn=reltool1.sfo01.justin.tv", "lsbdistcodename=precise"}

	fqdn, err := parseFQDN(tags)
	if err != nil {
		t.Fatal(err)
	}

	if fqdn != "reltool1.sfo01.justin.tv" {
		t.Fatalf("expected: reltool1.sfo01.justin.tv, got: %v", fqdn)
	}
}

// TestGetAliveHostnames is a first swipe at adding tests for GetAliveHostnames.
// Due to the difficulty of faking consul this is actually testing a bunch of things:
//
// 1. A datacenter that doesn't have a leader (returns 500).
// 2. A datacenter that has a server that is alive.
// 3. A datacenter that has a server that is not alive.
func TestGetAliveHostnames(t *testing.T) {
	ts := fakeConsul()
	u, err := url.Parse(ts.URL)
	if err != nil {
		t.Fatal(err)
	}

	client, err := NewClient(u.Host, nil)
	if err != nil {
		t.Fatal(err)
	}

	hosts, err := GetAliveHostnames(client, []string{"nodeinfo", "nodeinfo2"}, "", true)
	if err != nil {
		t.Fatal(err)
	}

	expected := "serf-1-app-0.dev.test1.justin.tv"
	t.Logf("Hosts: %#v", hosts)
	if len(hosts) != 1 || (len(hosts) > 0 && hosts[0] != expected) {
		t.Errorf(`GetAliveHostnames("nodeinfo", "nodeinfo2") = %#v; want: %#v`, hosts, []string{expected})
	}

	defer ts.Close()
}

func TestGetAliveHostnamesWithDCs(t *testing.T) {
	ts := fakeConsulDCs()
	u, err := url.Parse(ts.URL)
	if err != nil {
		t.Fatal(err)
	}

	client, err := NewClient(u.Host, nil)
	if err != nil {
		t.Fatal(err)
	}

	DChosts, err := GetAliveHostnamesWithDCs(client, []string{"nodeinfo", "nodeinfo2", "nodeinfo3"}, "", true)
	if err != nil {
		t.Fatal(err)
	}

	expected1 := DCtoNodes{DC: "test1", Nodes: []string{"serf-1-app-0.dev.test1.justin.tv"}}
	expected2 := DCtoNodes{DC: "test2", Nodes: []string{"other-app-0.dev.test2.justin.tv"}}
	expected := []DCtoNodes{expected1, expected2}
	expectedJson, _ := json.Marshal(expected)
	receivedJson, _ := json.Marshal(DChosts)
	//go doesn't have map equivalence w/o reflect package, or rolling your own
	if !(reflect.DeepEqual(DChosts, expected)) {
		t.Errorf(`GetAliveHostnamesDCs("nodeinfo", "nodeinfo2", "nodeinfo3") = %#v; want: %#v`, string(receivedJson), string(expectedJson))
	}

	defer ts.Close()
}
