package main

import (
	"bytes"
	"io"
	"log"
	"net/http"
	"net/http/httputil"
	"net/url"
	"strings"

	"github.com/gorilla/mux"
)

type router struct {
	mux     http.Handler
	grafana *url.URL
}

func (rt *router) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	rt.getHandler(r)(w, r)
}

func (rt *router) getHandler(r *http.Request) http.HandlerFunc {
	const (
		goGet         = "go-get"
		grafanaProxy  = "grafana-proxy"
		grafanaPublic = "/public/dashboards/"
	)

	if r.URL.Query().Get(goGet) == "1" {
		// We handle requests from the `go get` command first, by returning a
		// page with particular meta tags set.
		return rt.mux.ServeHTTP
	}

	if r.URL.Query().Get(grafanaProxy) == "1" {
		// This request is from the second phase of the dashboard load
		// process. The browser has already loaded the main page, and is now
		// requesting the grafana iframe.
		return rt.ServeGrafanaProxy
	}

	if strings.HasPrefix(r.URL.Path, grafanaPublic) {
		// This request is from the third phase of the dashboard load process.
		// The browser has the main page an iframe, and is now requesting the
		// javascript file describing the dashboard to display.
		r.URL.Path = "/static/" + strings.TrimPrefix(r.URL.Path, grafanaPublic)
	}
	if r.URL.Path == "/static/graph.js" {
		return rt.mux.ServeHTTP
	}

	if ref, err := url.Parse(r.Referer()); err == nil && ref.Query().Get(grafanaProxy) == "1" {
		// The dashboard structure has loaded, the browser is now requesting
		// data and other static assets from grafana. We proxy these requests
		// without modification.
		return rt.ServeGrafanaProxy
	}

	return rt.mux.ServeHTTP
}

func (rt *router) ServeGrafanaProxy(w http.ResponseWriter, r *http.Request) {
	httputil.NewSingleHostReverseProxy(rt.grafana).ServeHTTP(w, r)
}

func Handlers(grafana *url.URL) http.Handler {
	r := mux.NewRouter()
	r.HandleFunc("/", BaseHandler)
	r.PathPrefix("/static/").Handler(
		http.StripPrefix("/static/",
			http.FileServer(http.Dir("./static"))))
	r.HandleFunc("/{owner}/{repo}", BaseHandler)
	r.HandleFunc("/{owner}/{repo}/", BaseHandler)
	r.HandleFunc("/{owner}/{repo}/{misc}", BaseHandler)
	r.PathPrefix("/{owner}/{repo}/{misc}/").HandlerFunc(BaseHandler)

	return &router{mux: r, grafana: grafana}
}

func BaseHandler(w http.ResponseWriter, req *http.Request) {
	var buf bytes.Buffer

	v := NewVanity(req)
	if err := templates.ExecuteTemplate(&buf, "base.html", v); err != nil {
		log.Printf("Error rendering template %q: %v", "base.html", err)
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	_, err := io.Copy(w, &buf)
	if err != nil {
		log.Printf("Error sending template %q: %v", "base.html", err)
	}
}
