package main

import (
	"bufio"
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strings"
	"testing"
)

func TestGoGet(t *testing.T) {
	gs := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.Header().Set("Is-Grafana", "1")
	}))
	defer gs.Close()

	grafana, err := url.Parse(gs.URL)
	if err != nil {
		t.Fatalf("grafana stub server has invalid url: %v", err)
	}

	ts := httptest.NewServer(Handlers(grafana))
	defer ts.Close()

	for _, tt := range []struct {
		uri     string
		referer string

		acceptor func(*http.Response) error
	}{
		{uri: "/?go-get=1",
			acceptor: requireGoGet(false)},
		{uri: "/foo/bar"},
		{uri: "/foo/bar?go-get=0"},
		{uri: "/foo/bar?go-get=1",
			acceptor: requireGoGet(true)},

		{uri: "/foo/bar?grafana-proxy=0",
			acceptor: rejectGrafana},
		{uri: "/foo/bar?grafana-proxy=1",
			acceptor: requireGrafana},
		{uri: "/foo/bar", referer: "/foo/bar",
			acceptor: rejectGrafana},
		{uri: "/foo/bar", referer: "/foo/bar?grafana-proxy=0",
			acceptor: rejectGrafana},
		{uri: "/foo/bar", referer: "/foo/bar?grafana-proxy=1",
			acceptor: requireGrafana},
		{uri: "/public/dashboards/graph.js", referer: "/foo/bar",
			acceptor: requireJS},

		{uri: "/foo/bar/"},
		{uri: "/foo/bar/foobar"},
		{uri: "/foo/bar/foobar/"},
		{uri: "/foo/bar/foobar/baz/"},
	} {
		fullURL := ts.URL + tt.uri

		t.Logf("Testing GoGet for %q", fullURL)

		req, err := http.NewRequest("GET", fullURL, nil)
		if err != nil {
			t.Errorf("could not create request: %v", err)
			continue
		}
		req.Header.Set("Referer", tt.referer)

		resp, err := http.DefaultClient.Do(req)
		if err != nil {
			t.Errorf("Failed to get %q", tt.uri)
			continue
		}
		defer resp.Body.Close()

		if resp.StatusCode != http.StatusOK {
			t.Errorf("Failed to get %q; Got status %d", tt.uri, resp.StatusCode)
			continue
		}

		if tt.acceptor != nil {
			err := tt.acceptor(resp)
			if err != nil {
				t.Errorf("response rejected for uri %q with referer %q: %v", tt.uri, tt.referer, err)
				continue
			}
		}
	}
}

func rejectGrafana(resp *http.Response) error {
	if resp.Header.Get("Is-Grafana") != "" {
		return fmt.Errorf("request incorrectly proxied to grafana")
	}
	return nil
}

func requireGrafana(resp *http.Response) error {
	if resp.Header.Get("Is-Grafana") != "1" {
		return fmt.Errorf("grafana request was not proxied")
	}
	return nil
}

func requireJS(resp *http.Response) error {
	switch contentType := resp.Header.Get("Content-Type"); contentType {
	case "application/javascript":
	case "application/x-javascript":
	default:
		return fmt.Errorf("non-javascript response: %v", contentType)
	}
	return nil
}

func requireGoGet(shouldHave bool) func(*http.Response) error {
	return func(resp *http.Response) error {
		var (
			haveGoImport bool
			haveGoSource bool
		)

		sc := bufio.NewScanner(resp.Body)
		for sc.Scan() {
			if strings.Contains(sc.Text(), "go-import") {
				haveGoImport = true
			}
			if strings.Contains(sc.Text(), "go-source") {
				haveGoSource = true
			}
		}

		if err := sc.Err(); err != nil {
			return err
		}

		if shouldHave {
			if !haveGoImport {
				return fmt.Errorf("did not find go-import tag")
			}
			if !haveGoSource {
				return fmt.Errorf("did not find go-source tag")
			}
		} else {
			if haveGoImport {
				return fmt.Errorf("did find go-import tag")
			}
			if haveGoSource {
				return fmt.Errorf("did find go-source tag")
			}
		}

		return nil
	}
}
