package main

import (
	"fmt"
	"html/template"
	"net/http"
	"net/url"
	"path"

	"github.com/gorilla/mux"
)

// Vanity represents the state and info needed to render a page.
type Vanity struct {
	Owner    string // Github owner name
	Repo     string // Github repo name
	FullName string

	GoGet      bool
	SiteName   string
	ImportName string
	RepoURL    template.URL
	VCS        string
	branch     string
}

func NewVanity(req *http.Request) *Vanity {
	vars := mux.Vars(req)

	owner, repo := vars["owner"], vars["repo"]

	vanity := &Vanity{
		Owner:    owner,
		Repo:     repo,
		FullName: path.Join(owner, repo),
		SiteName: siteName,
		VCS:      "git",
	}

	vanity.RepoURL = template.URL(repoURL(vanity.Owner, vanity.Repo))
	vanity.ImportName = importName(vanity.Owner, vanity.Repo)
	vanity.branch = "master"

	query := req.URL.Query()
	if query.Get("go-get") != "" && req.URL.Path != "/" {
		vanity.GoGet = true
	}

	return vanity
}

func (v *Vanity) Title() string {
	return fmt.Sprintf("%s/%s - %s", v.Owner, v.Repo, v.SiteName)
}

func (v *Vanity) GithubURL() template.URL {
	u := &url.URL{
		Scheme: "https",
		Host:   "git.xarth.tv",
		Path:   path.Join(v.Owner, v.Repo),
	}
	return template.URL(u.String())
}

func (v *Vanity) ProjectURL() template.URL {
	return v.GithubURL()
}

func (v *Vanity) DirTemplate() template.URL {
	return v.GithubURL() + template.URL("/tree/"+v.branch+"{/dir}")
}

func (v *Vanity) FileTemplate() template.URL {
	return v.GithubURL() + template.URL("/blob/"+v.branch+"{/dir}/{file}#L{line}")
}

func (v *Vanity) DeployURL() template.URL {
	u := &url.URL{
		Scheme:   "https",
		Host:     "deploy.xarth.tv",
		Fragment: path.Join("/", v.Owner, v.Repo),
	}
	return template.URL(u.String())
}
