variable "profile" {
  description = "aws credential profile"
}

variable "region" {
  description = "aws region"
}

variable "service_name" {
  description = "name of service"
  default     = "villagers"
}

variable "image_tag" {
  description = "revision of container to deploy"
}

variable "memory_mb" {
  description = "required memory reservation for container"
}

variable "desired_count" {
  description = "number of containers to deploy"
}

provider "aws" {
  region  = "${var.region}"
  profile = "${var.profile}"
}

# Remote data from upstream terraform. This is used to fetch the target ECS cluster for ops accounts.
data "terraform_remote_state" "cluster_region" {
  backend = "s3"

  config {
    profile = "${var.profile}"
    region  = "us-west-2"
    bucket  = "${var.profile}-tf-state"
    key     = "cluster/${var.region}/terraform.tfstate"
  }
}

# Creates a new task definition which contains metadata about running containers.
#
# We always create a new task definition (even if it corresponds to an older
# version of the application) because task definitions are versioned by a
# monotonically increasing integer; there is no easy way to query
# for the task definition corresponding to app version X.
resource "aws_ecs_task_definition" "task_definition" {
  family = "${var.service_name}"

  container_definitions = <<EOF
[
  {
    "name": "${var.service_name}",
    "memory": ${var.memory_mb},
    "image": "docker-registry.internal.justin.tv/release/villagers:${var.image_tag}",
    "essential": true,
    "portMappings": [
      {
        "containerPort": 3498,
        "hostPort": 3498,
        "protocol": "tcp"
      },
      {
        "containerPort": 8943,
        "hostPort": 8943,
        "protocol": "udp"
      }
    ],
    "logConfiguration": {
      "logDriver": "awslogs",
      "options": {
        "awslogs-group": "ecs_${var.service_name}",
        "awslogs-region": "${var.region}",
        "awslogs-stream-prefix": "${var.service_name}"
      }
    }
  }
]
EOF
}

# Creates an ECS service, comprised of N running tasks (containers) defined via the task definition above.
# The deployment_maximum_percent must be <= 100% due to its distinctInstance placement constraint.
resource "aws_ecs_service" "service" {
  name                               = "${var.service_name}"
  cluster                            = "${data.terraform_remote_state.cluster_region.target_cluster}"
  task_definition                    = "${aws_ecs_task_definition.task_definition.arn}"
  desired_count                      = "${var.desired_count}"
  deployment_maximum_percent         = "100"
  deployment_minimum_healthy_percent = "50"

  placement_constraints {
    type = "distinctInstance"
  }
}

# Creates a log group that contains the STDOUT of the villagers containers.
resource "aws_cloudwatch_log_group" "group" {
  name = "ecs_${var.service_name}"
}

# Creates a cloudwatch alarm for the number of healthy tasks in the service.
# This number will flap if containers fail healthchecks.
#
# See: http://docs.aws.amazon.com/AmazonECS/latest/developerguide/cloudwatch-metrics.html#cw_running_task_count
resource "aws_cloudwatch_metric_alarm" "healthy_task_count" {
  alarm_name          = "ecs_task_${var.service_name}_healthy_count"
  alarm_description   = "Monitors number of healthy tasks for ${var.service_name}"
  comparison_operator = "LessThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "CPUUtilization"
  namespace           = "AWS/ECS"
  period              = "60"
  statistic           = "SampleCount"
  threshold           = "${var.desired_count}"

  dimensions {
    ServiceName = "${var.service_name}"
    ClusterName = "${data.terraform_remote_state.cluster_region.target_cluster}"
  }
}

# Creates a cloudwatch alarm for the CPU utilization of the service.
resource "aws_cloudwatch_metric_alarm" "service_cpu_utilization" {
  alarm_name          = "ecs_task_${var.service_name}_cpu_utilization"
  alarm_description   = "Monitors CPU utilization for ${var.service_name}"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "CPUUtilization"
  namespace           = "AWS/ECS"
  period              = "120"
  statistic           = "Average"
  threshold           = "60"

  dimensions {
    ServiceName = "${var.service_name}"
    ClusterName = "${data.terraform_remote_state.cluster_region.target_cluster}"
  }
}

# Creates a cloudwatch alarm for the memory utilization for the service.
resource "aws_cloudwatch_metric_alarm" "service_memory_utilization" {
  alarm_name          = "ecs_task_${var.service_name}_memory_utilization"
  alarm_description   = "Monitors memory utilization for ${var.service_name}"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "MemoryUtilization"
  namespace           = "AWS/ECS"
  period              = "120"
  statistic           = "Average"
  threshold           = "60"

  dimensions {
    ServiceName = "${var.service_name}"
    ClusterName = "${data.terraform_remote_state.cluster_region.target_cluster}"
  }
}

output "image_tag" {
  value = "${var.image_tag}"
}
