// Copyright (c) 2012, Suryandaru Triandana <syndtr@gmail.com>
// All rights reserved.
//
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package ldb

import "bytes"

type bytesComparer struct{}

func (*bytesComparer) Compare(a, b []byte) int {
	return bytes.Compare(a, b)
}

func (*bytesComparer) Name() string {
	return "leveldb.BytewiseComparator"
}

func (*bytesComparer) Separator(dst, a, b []byte) []byte {
	i, n := 0, len(a)
	if n > len(b) {
		n = len(b)
	}
	for ; i < n && a[i] == b[i]; i++ {
	}
	if i >= n {
		// Do not shorten if one string is a prefix of the other
	} else if c := a[i]; c < 0xff && c+1 < b[i] {
		dst = append(dst, a[:i+1]...)
		dst[i]++
		return dst
	}
	return nil
}

func (*bytesComparer) Successor(dst, b []byte) []byte {
	for i, c := range b {
		if c != 0xff {
			dst = append(dst, b[:i+1]...)
			dst[i]++
			return dst
		}
	}
	return nil
}

// FastBytewiseComparator is a faster version of the default LevelDB bytewise
// comparer.
//
// Its methods are attached to the struct pointer, which provides a
// substantial speedup over the default comparer by removing the autogenerated
// wrapper method from the call stack.
var FastBytewiseComparator = new(bytesComparer)
