package tv.justin.rockpaperscissors;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.handlers.AsyncHandler;
import com.amazonaws.services.kinesis.AmazonKinesisAsync;
import com.amazonaws.services.kinesis.model.PutRecordRequest;
import com.amazonaws.services.kinesis.model.PutRecordResult;
import com.google.protobuf.ByteString;
import java.nio.ByteBuffer;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nonnull;
import javax.xml.bind.DatatypeConverter;
import org.apache.commons.codec.binary.Hex;
import tv.justin.rockpaperscissors.proto.EventOuterClass;

public class EventPublisher {

  private final static Logger LOG = Logger.getLogger(EventPublisher.class.getName());

  private final AmazonKinesisAsync kinesisClient;
  private final String streamName;

  public EventPublisher(@Nonnull AmazonKinesisAsync kinesisClient, @Nonnull String streamName) {
      this.kinesisClient = kinesisClient;
      this.streamName = streamName;
  }

  public void publish(@Nonnull final EventOuterClass.Event event) {
      ByteBuffer data = ByteBuffer.wrap(event.toByteArray());
      String partitionKey = Hex.encodeHexString(event.getUuid().toByteArray());

      // TODO: handle failures and retrying
      kinesisClient.putRecordAsync(streamName, data, partitionKey, new AsyncHandler<PutRecordRequest, PutRecordResult>() {
          @Override
          public void onSuccess(PutRecordRequest request, PutRecordResult result) {
              String encodedUUID = DatatypeConverter.printBase64Binary(event.getUuid().toByteArray());
              LOG.log(Level.INFO, "Published {0} event to RockPaperScissors with UUID {1}",
                  new Object[]{event.getType(), encodedUUID});
          }

          @Override
          public void onError(Exception e) {
              LOG.log(Level.WARNING, "Failed to publish event to RockPaperScissors: " + e.getMessage(), e);
          }
      });
  }

  @Nonnull
  public static EventOuterClass.Event createEvent(double timestamp, @Nonnull String eventType, @Nonnull ByteString body, @Nonnull Map<String,String> attributes) {
      // TODO: check eventType isn't empty

      UUID uuid = UUID.randomUUID();
      ByteBuffer bb = ByteBuffer.wrap(new byte[16]);
      bb.putLong(uuid.getMostSignificantBits());
      bb.putLong(uuid.getLeastSignificantBits());
      bb.rewind();
      ByteString uuidByteString = ByteString.copyFrom(bb);

      EventOuterClass.Event.Builder eventBuilder = EventOuterClass.Event.newBuilder()
          .setUuid(uuidByteString)
          .setTimestamp(timestamp)
          .setType(eventType)
          .setBody(body);

      if (attributes != null) {
          for (Map.Entry<String, String> entry : attributes.entrySet()) {
              eventBuilder.addAttributes(
                  EventOuterClass.Event.Attribute.newBuilder()
                      .setKey(entry.getKey())
                      .setValue(entry.getValue())
                      .build());
          }
      }

      return eventBuilder.build();
  }
}
