package tv.justin.rockpaperscissors;

import com.amazonaws.SdkClientException;
import com.amazonaws.auth.profile.ProfileCredentialsProvider;
import com.amazonaws.auth.profile.ProfilesConfigFile;
import com.amazonaws.services.kinesis.AmazonKinesisAsync;
import com.amazonaws.services.kinesis.AmazonKinesisAsyncClientBuilder;
import com.cloudbees.plugins.credentials.CredentialsProvider;
import com.cloudbees.plugins.credentials.CredentialsMatcher;
import com.cloudbees.plugins.credentials.CredentialsMatchers;
import com.cloudbees.plugins.credentials.common.StandardListBoxModel;
import com.cloudbees.plugins.credentials.domains.DomainRequirement;
import hudson.Extension;
import hudson.Util;
import hudson.model.Descriptor.FormException;
import hudson.security.ACL;
import hudson.util.ListBoxModel;
import java.io.InputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import jenkins.model.GlobalConfiguration;
import jenkins.model.Jenkins;
import net.sf.json.JSONObject;
import org.jenkinsci.plugins.plaincredentials.FileCredentials;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.DataBoundSetter;
import org.kohsuke.stapler.StaplerRequest;

@Extension
public class RockPaperScissorsConfiguration extends GlobalConfiguration {

    private final static Logger LOG = Logger.getLogger(RockPaperScissorsPublisher.class.getName());

    private String awsProfileCredentialId;
    private String awsProfileName;
    private String region;
    private String streamName;
    private boolean publishBuildStats = true;
    private boolean publishTestResults = true;
    private boolean publishTestCoverage = true;

    @Nullable private transient EventPublisher eventPublisher;

    @DataBoundConstructor
    public RockPaperScissorsConfiguration(String awsProfileCredentialId, String awsProfileName, String region, String streamName, boolean publishBuildStats, boolean publishTestResults, boolean publishTestCoverage) {
        super();
        this.awsProfileCredentialId = awsProfileCredentialId;
        this.awsProfileName = awsProfileName;
        this.region = region;
        this.streamName = streamName;
        this.publishBuildStats = publishBuildStats;
        this.publishTestResults = publishTestResults;
        this.publishTestCoverage = publishTestCoverage;
    }

    public RockPaperScissorsConfiguration() {
        super();
        load();
    }

    public String getAwsProfileCredentialId() {
        return awsProfileCredentialId;
    }

    @DataBoundSetter
    public void setAwsProfileCredentialId(String awsProfileCredentialId) {
        this.awsProfileCredentialId = Util.fixEmpty(awsProfileCredentialId);
    }

    @Nonnull
    public ListBoxModel doFillAwsProfileCredentialIdItems() {
        Jenkins instance = Jenkins.getInstance();
        ACL rootACL = instance.getACL();
        if (!rootACL.hasPermission(Jenkins.ADMINISTER)) {
            return new StandardListBoxModel().includeCurrentValue(awsProfileCredentialId);
        }
        return new StandardListBoxModel()
                .includeEmptyValue()
                .includeMatchingAs(ACL.SYSTEM,
                                   Jenkins.getInstance(),
                                   FileCredentials.class,
                                   Collections.<DomainRequirement>emptyList(),
                                   CredentialsMatchers.always())
                .includeCurrentValue(awsProfileCredentialId);
    }

    public String getAwsProfileName() {
        return awsProfileName;
    }

    @DataBoundSetter
    public void setAwsProfileName(String awsProfileName) {
        this.awsProfileName = Util.fixEmpty(awsProfileName);
    }

    public String getRegion() {
        return region;
    }

    @DataBoundSetter
    public void setRegion(@Nonnull String region) {
        this.region = Util.fixEmpty(region);
    }

    public String getStreamName() {
        return streamName;
    }

    @DataBoundSetter
    public void setStreamName(@Nonnull String streamName) {
        this.streamName = Util.fixEmpty(streamName);
    }

    public boolean isPublishBuildStats() {
        return publishBuildStats;
    }

    @DataBoundSetter
    public void setPublishBuildStats(boolean publishBuildStats) {
        this.publishBuildStats = publishBuildStats;
    }

    public boolean isPublishTestResults() {
        return publishTestResults;
    }

    @DataBoundSetter
    public void setPublishTestResults(boolean publishTestResults) {
        this.publishTestResults = publishTestResults;
    }

    public boolean isPublishTestCoverage() {
        return publishTestCoverage;
    }

    @DataBoundSetter
    public void setPublishTestCoverage(boolean publishTestCoverage) {
        this.publishTestCoverage = publishTestCoverage;
    }

    @Override
    public boolean configure(@Nonnull final StaplerRequest req, @Nonnull final JSONObject formData) throws FormException {
        req.bindJSON(this, formData);
        eventPublisher = null;
        save();
        return super.configure(req, formData);
    }

    @CheckForNull
    public static RockPaperScissorsConfiguration get() {
        return RockPaperScissorsConfiguration.all().get(RockPaperScissorsConfiguration.class);
    }

    @CheckForNull
    public EventPublisher getEventPublisher() {
      if (eventPublisher == null) {
        loadEventPublisher();
      }
      return eventPublisher;
    }

    private void loadEventPublisher() {
      eventPublisher = null;

      if (getAwsProfileCredentialId() == null || getAwsProfileCredentialId().isEmpty()) {
        LOG.warning("RockPaperScissors event publishing disabled, needs credentials configuration.");
        return;
      }

      ProfilesConfigFile profilesConfigFile;
      try {
          profilesConfigFile = getProfilesConfigFile();
      }
      catch (IOException e) {
          LOG.log(Level.WARNING, "Could not get configure AWS credentials profile", e);
          return;
      }
      catch (SdkClientException e) {
        LOG.log(Level.WARNING, "Could not get load AWS credentials profile", e);
        return;
      }
      ProfileCredentialsProvider credentialsProvider = new ProfileCredentialsProvider(profilesConfigFile, getAwsProfileName());
      AmazonKinesisAsync kinesisClient = AmazonKinesisAsyncClientBuilder.standard()
          .withRegion(getRegion())
          .withCredentials(credentialsProvider)
          .build();

      // TODO: prevent streamName from being null
      eventPublisher = new EventPublisher(kinesisClient, getStreamName());
    }

    @CheckForNull
    private static FileCredentials lookupCredentials(@Nonnull String credentialId) {
        List<FileCredentials> credentials = CredentialsProvider.lookupCredentials(
            FileCredentials.class, Jenkins.getInstance(), ACL.SYSTEM, Collections.<DomainRequirement>emptyList());
        CredentialsMatcher matcher = CredentialsMatchers.withId(credentialId);
        return CredentialsMatchers.firstOrNull(credentials, matcher);
    }

    @Nonnull
    private ProfilesConfigFile getProfilesConfigFile() throws IOException, SdkClientException {
      FileCredentials profileCredentials = lookupCredentials(getAwsProfileCredentialId());
      if (profileCredentials == null) {
        return new ProfilesConfigFile();
      }

      InputStream in = profileCredentials.getContent();
      try {
          Path tempFilePath = Files.createTempFile("rockpaperscissors", ".aws_credentials");
          try {
              Files.copy(in, tempFilePath, StandardCopyOption.REPLACE_EXISTING);
              ProfilesConfigFile profilesConfigFile = new ProfilesConfigFile(tempFilePath.toString());
              return profilesConfigFile;
          }
          finally {
              try {
                Files.delete(tempFilePath);
              }
              catch (IOException e) {
                LOG.log(Level.WARNING, "Failed to delete temp file", e);
              }
          }
      }
      finally {
          in.close();
      }
    }

}
