package tv.justin.rockpaperscissors;

import com.google.protobuf.ByteString;
import hudson.FilePath;
import hudson.scm.ChangeLogSet;
import hudson.scm.ChangeLogSet.Entry;
import hudson.model.AbstractBuild;
import hudson.model.AbstractProject;
import hudson.model.Cause;
import hudson.model.Result;
import hudson.model.Run.Artifact;
import hudson.model.TaskListener;
import hudson.model.User;
import hudson.plugins.git.Branch;
import hudson.plugins.git.Revision;
import hudson.plugins.git.util.BuildData;
import hudson.security.ACL;
import hudson.tasks.junit.SuiteResult;
import hudson.tasks.junit.TestResult;
import hudson.tasks.junit.TestResultAction;
import java.io.InputStream;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;
import jenkins.metrics.impl.TimeInQueueAction;
import jenkins.model.Jenkins;
import jenkins.util.VirtualFile;
import org.apache.commons.lang.StringUtils;
import org.jenkinsci.plugins.plaincredentials.FileCredentials;
import tv.justin.rockpaperscissors.proto.EventOuterClass;
import tv.justin.rockpaperscissors.proto.ProjectMetadataOuterClass;
import tv.justin.rockpaperscissors.proto.events.JenkinsBuild;

public class RockPaperScissorsPublisher {

    private final static Logger LOG = Logger.getLogger(RockPaperScissorsPublisher.class.getName());
    // "git@git-aws.internal.justin.tv:dta/rockpaperscissors.git"
    // "https://github.com/twitchscience/rs_ingester.git"
    private final static Pattern GITURLPATTERN = Pattern.compile(
        "^(?:git+ssh|https?://)?(?:[^@]+@)?(?<host>[^:/]+)[:/](?<name>[^\\.]+)(?:\\.git)?$");

    public RockPaperScissorsPublisher() {
    }

    public void onCompleted(@Nonnull AbstractBuild build, @Nonnull TaskListener listener) {
        RockPaperScissorsConfiguration config = RockPaperScissorsConfiguration.get();
        if (config == null) {
            LOG.warning("Could not get RockPaperScissorsConfiguration.");
            return;
        }

        if (build == null) {
            LOG.warning("Can't publish build stats for null build.");
            return;
        }

        EventPublisher publisher = config.getEventPublisher();
        if (publisher == null) {
            LOG.warning("RockPaperScissors can't publish due to misconfiguration or network issue.");
            return;
        }

        JenkinsBuild.JenkinsBuildStats buildStats = getBuildStats(build);
        Map<String, String> attributes = getEventAttributes(buildStats);

        if (config.isPublishBuildStats()) {
            publishBuildStatsEvent(publisher, buildStats, attributes);
        }

        if (config.isPublishTestResults()) {
            publishTestResultsEvent(publisher, build, attributes);
        }

        if (config.isPublishTestCoverage()) {
            publishTestCoverageEvent(publisher, build, attributes);
        }
    }

    @Nonnull
    private JenkinsBuild.JenkinsBuildStats getBuildStats(@Nonnull AbstractBuild build) {
        Set<String> causeStrings = new HashSet<String>();
        @SuppressWarnings("unchecked") List<Cause> causes = build.getCauses();
        for (Cause cause : causes) {
            causeStrings.add(cause.getShortDescription());
        }

        JenkinsBuild.JenkinsBuildStats.Result buildResult = JenkinsBuild.JenkinsBuildStats.Result.NO_RESULT;
        Result jenkinsBuildResult = build.getResult();
        if (jenkinsBuildResult != null) {
            buildResult = JenkinsBuild.JenkinsBuildStats.Result.forNumber(jenkinsBuildResult.ordinal);
        }

        JenkinsBuild.JenkinsBuildStats.Builder buildStatsBuilder =
            JenkinsBuild.JenkinsBuildStats.newBuilder()
            // "537"
            .setBuildId(build.getId())
            // "https://jenkins.internal.justin.tv/job/dta-rockpaperscissors/537/"
            .setUrl(getRootUrl() + build.getUrl())
            .setStartTimeInMillis(build.getStartTimeInMillis())
            .setDurationInMillis(build.getDuration())
            // "jenkins-prod-slave-3"
            .setBuiltOn(build.getBuiltOnStr())
            // ABORTED, FAILURE, NOT_BUILT, SUCCESS, UNSTABLE
            .setResult(buildResult)
            // "Started by GitHub push by gribilly"
            .setCause(StringUtils.join(causeStrings, ", "));

        AbstractProject project = build.getProject();
        if (project != null) {
            // "dta-rockpaperscissors"
            buildStatsBuilder.setJobName(project.getDisplayName());
        }
        else {
            LOG.warning("Couldn't get project for build.");
        }

        BuildData gitBuildData = build.getAction(BuildData.class );
        if (gitBuildData != null) {
            Set<String> remoteUrls = gitBuildData.getRemoteUrls();
            if (remoteUrls != null && !remoteUrls.isEmpty()) {
                // "git@git-aws.internal.justin.tv:dta/rockpaperscissors.git"
                String remoteUrl = remoteUrls.iterator().next();
                ProjectMetadataOuterClass.GitHubRepository gitHubRepository =
                    buildGitHubRepositoryFromUrl(remoteUrl);
                buildStatsBuilder.setGithubRepository(gitHubRepository);
            }
            else {
                LOG.warning("Couldn't get Git remote URLs for build.");
            }

            Revision revision = gitBuildData.getLastBuiltRevision();
            if (revision != null) {
                // "5a69cb0b8f6d6a4de67eb08f8b0dc9223daed152"
                buildStatsBuilder.setCommitId(revision.getSha1String());

                Collection<Branch> branches = revision.getBranches();
                if (branches != null && !branches.isEmpty()) {
                    // "origin/master" or "refs/remotes/origin/master"
                    Branch branch = branches.iterator().next();

                    // "master"
                    String checkoutBranch = branch.getName();
                    checkoutBranch = checkoutBranch.replaceFirst("^refs/remotes/", "");
                    checkoutBranch = checkoutBranch.replaceFirst("^origin/", "");
                    buildStatsBuilder.setBranch(checkoutBranch);
                }
                else {
                    LOG.warning("Couldn't get Git branches for build.");
                }
            }
            else {
                LOG.warning("Couldn't get Git revision for build.");
            }
        }
        else {
            LOG.warning("Couldn't get Git build data for build.");
        }

        ChangeLogSet<?> changeSet = build.getChangeSet();
        for (Entry entry : changeSet) {
            buildStatsBuilder.addChangeSet(
                JenkinsBuild.JenkinsBuildStats.ChangeSet.newBuilder()
                .setCommitId(entry.getCommitId())
                .setTimestamp(entry.getTimestamp()));
        }

        TimeInQueueAction timeInQueueAction = build.getAction(TimeInQueueAction.class );
        if (timeInQueueAction != null) {
            buildStatsBuilder.setQueuingDurationMillis(timeInQueueAction.getQueuingDurationMillis());
        }

        return buildStatsBuilder.build();
    }

    private void publishBuildStatsEvent(@Nonnull EventPublisher publisher,
                                        @Nonnull JenkinsBuild.JenkinsBuildStats buildStats,
                                        @Nonnull Map<String, String> attributes) {

        EventOuterClass.Event event = EventPublisher.createEvent(
            buildStats.getStartTimeInMillis() / 1000.0, "JenkinsBuildStats",
            buildStats.toByteString(), attributes);
        publisher.publish(event);
    }

    private void publishTestResultsEvent(@Nonnull EventPublisher publisher, @Nonnull AbstractBuild build,
                                         @Nonnull Map<String, String> attributes) {
        Set<String> testReportFilenames = new HashSet<String>();

        TestResultAction testResultAction = build.getAction(TestResultAction.class );
        if (testResultAction != null) {
            TestResult testResult = testResultAction.getResult();
            for (SuiteResult suite : testResult.getSuites()) {
                testReportFilenames.add(suite.getFile());
            }
        }

        for (String testReportFilename : testReportFilenames) {
            FilePath testReportFile = new FilePath(build.getWorkspace(), testReportFilename);
            VirtualFile testReport = testReportFile.toVirtualFile();
            try {
                if (testReport.isFile()) {
                    InputStream in = testReport.open();
                    ByteString body = ByteString.readFrom(in);
                    EventOuterClass.Event event = EventPublisher.createEvent(
                        build.getStartTimeInMillis() / 1000.0,
                        "JenkinsTestResults", body, attributes);
                    publisher.publish(event);
                } else {
                    LOG.warning(String.format("Test report missing: %s", testReportFilename));
                }
            }
            catch (IOException e) {
                LOG.log(Level.WARNING,
                        String.format("An exception occurred while reading %s", testReportFilename), e);
            }
        }
    }

    private void publishTestCoverageEvent(@Nonnull EventPublisher publisher, @Nonnull AbstractBuild build,
                                          @Nonnull Map<String, String> attributes) {
        VirtualFile artifactRoot = build.getArtifactManager().root();
        String[] coverageFiles = {"coverage.xml", "codecov_coverage.json"};
        for(String coverageFile : coverageFiles) {
            VirtualFile testCoverage = artifactRoot.child(coverageFile);
            try {
                if (testCoverage.isFile()) {
                    InputStream in = testCoverage.open();
                    ByteString body = ByteString.readFrom(in);

                    Map<String, String> coverageAttributes = new HashMap<String, String>(attributes);
                    coverageAttributes.put("coverage_file", coverageFile);

                    EventOuterClass.Event event = EventPublisher.createEvent(
                        build.getStartTimeInMillis() / 1000.0,
                        "JenkinsTestCoverage", body, coverageAttributes);
                    publisher.publish(event);
                }
            }
            catch (IOException e) {
                LOG.log(Level.WARNING,
                        String.format("An exception occurred while reading %s", coverageFile), e);
            }
        }
    }

    @Nonnull
    private static ProjectMetadataOuterClass.GitHubRepository buildGitHubRepositoryFromUrl(
        @Nonnull String gitUrl) {
        Matcher m = GITURLPATTERN.matcher(gitUrl);

        ProjectMetadataOuterClass.GitHubRepository.Builder gitHubRepositoryBuilder =
            ProjectMetadataOuterClass.GitHubRepository.newBuilder();

        if (m.matches()) {
            // "git-aws.internal.justin.tv"
            gitHubRepositoryBuilder.setHost(m.group("host"));
            // "dta/rockpaperscissors"
            gitHubRepositoryBuilder.setName(m.group("name"));
        }
        else {
            LOG.warning(
                String.format("Git remote URL \"%s\" doesn't match expected format.", gitUrl));
        }

        return gitHubRepositoryBuilder.build();
    }

    @Nonnull
    private static Map<String, String> getEventAttributes(
        @Nonnull JenkinsBuild.JenkinsBuildStats buildStats) {
        Map<String, String> attributes = new HashMap<String, String>();

        if (!buildStats.getJobName().isEmpty()) {
            attributes.put("jenkins_job_name", buildStats.getJobName());
        }

        if (buildStats.hasGithubRepository()) {
            ProjectMetadataOuterClass.GitHubRepository gitHubRepository = buildStats.getGithubRepository();
            attributes.put("github_repository",
                           "https://" + gitHubRepository.getHost() + "/" + gitHubRepository.getName());
        }

        if (!buildStats.getBranch().isEmpty()) {
            attributes.put("github_branch", buildStats.getBranch());
        }

        return attributes;
    }

    @Nonnull
    private static String getRootUrl() {
        if (Jenkins.getInstance().getRootUrl() == null) {
            return "http://localhost:8080/";
        } else {
            return Jenkins.getInstance().getRootUrl();
        }
    }

}
