/* globals tmi */
import Component from 'ember-component';
import injectService from 'ember-service/inject';
import { assert } from 'ember-metal/utils';
import { bind, schedule } from 'ember-runloop';
import { htmlSafe } from 'ember-string';
import LinkedList from 'mobile-web-client/utils/linked-list';

const BOTTOM_SCROLL_THRESHOLD = 10;

export default Component.extend({
  classNames: ['chat-room'],

  badges: injectService(),

  channelId: null,
  channelName: null,
  keepBottomScrolled: true,
  messages: null,
  tmiClient: null,
  userColors: null,

  init() {
    this._super(...arguments);

    this.set('messages', new LinkedList());
    this.userColors = {};

    this._tmiConnectPromise = undefined;
  },

  didInsertElement() {
    this._super(...arguments);

    //this._scrollEvents = `scroll mousedown wheel DOMMouseScroll mousewheel keyup`; // TODO replace with mobile-specific scroll events

    //this.addEventListener(this.element, this._scrollEvents, (event) => {
    //  if (this.isDestroying || this.isDestroyed) { return; }

    //  if (event.which > 0 || event.type === 'mousedown' || event.type === 'mousewheel') { // TODO replace with mobile-specific scroll events
    //    let scrollDistanceToBottom = this.element.scrollHeight - this.element.scrollTop - this.element.offsetHeight;

    //    this.set('keepBottomScrolled', scrollDistanceToBottom <= BOTTOM_SCROLL_THRESHOLD);
    //  }
    //});

    this.setupRoom();
  },

  setupRoom() {
    let channelName = this.get('channelName');
    assert(channelName, 'A channel name must be provided to the `chat-room` component.');

    let tmiClient = tmi.client({
      connection: {
        secure: true
      },
      channels: [channelName]
    });

    this.set('tmiClient', tmiClient);

    tmiClient.on('chat', bind(this, this.addMessage));

    this.get('badges').loadBadges(this.get('channelId')).then(() => {
      if (this.isDestroying || this.isDestroyed) { return; }

      this._tmiConnectPromise = tmiClient.connect();
      return this._tmiConnectPromise;
    }).then(function () {
      /* eslint-disable no-console */
      console.timeEnd('MWE_CHAT_READY');
      /* eslint-enable no-console */
    });
  },

  addMessage(channel, userState, message) {
    if (this.isDestroyed || this.isDestroying) { return; }

    if (!userState['display-name']) {
      userState['display-name'] = userState.username;
    }

    this.get('messages').append(this.buildMessage(message, userState));

    requestAnimationFrame(() => {
      this.updateRender();
    });

    //this.throttleTask('updateRender', 25);
    // this.updateRender();
  },

  checkMessageCap() {
    while (this.messages.length > 200) {
      this.messages.removeFirst();
    }
  },

  buildMessage(message, userState) {
    return {
      badges: this.get('badges').parseBadges(this.get('channelId'), userState.badges),
      state: userState,
      text: this.parseMessage(message, userState.emotes)
    }
  },

  parseMessage(message, emotes) {
    if (!emotes || emotes.length === 0) { return message; }

    let newMessage = message.split('');

    for (let emoteIndex in emotes) {
      let emote = emotes[emoteIndex];

      for (let charIndexes in emote) {
        let emoteIndexes = emote[charIndexes];

        if (typeof(emoteIndexes) === 'string') {
          emoteIndexes = emoteIndexes.split('-');
          emoteIndexes = [parseInt(emoteIndexes[0]), parseInt(emoteIndexes[1])];

          for (let i = emoteIndexes[0]; i <= emoteIndexes[1]; ++i) {
            newMessage[i] = '';
          }

          newMessage[emoteIndexes[0]] = `<img class="emoticon" src="//static-cdn.jtvnw.net/emoticons/v1/${emoteIndex}/3.0">`;
        }
      }
    }
    // TODO: This is a naive implementation, we need to escape the non-emote parts of the message,
    // but that messes with parsing so not sure when to do it.
    return htmlSafe(newMessage.join(''));
  },

  updateRender() {
    this.checkMessageCap();
    this.messages.arrayContentDidChange();
    schedule('afterRender', this, this.scrollBottom);
  },

  scrollBottom() {
    if (this.get('keepBottomScrolled')) {
      this.element.scrollTop = this.element.scrollHeight;
    }
  },

  willDestroy() {
    let tmiClient = this.get('tmiClient');

    if (this._tmiConnectPromise) {
      this._tmiConnectPromise.then(() => {
        tmiClient.disconnect();
      });
    }

    this._super(...arguments);
  }
});
