import Mixin from 'ember-metal/mixin';
import run from 'ember-runloop';
import { assert } from 'ember-metal/utils';

/**
 ContextBoundEventListenersMixin provides a mechanism to attach event listeners
 with runloops and automatic removal when the host object is removed from DOM.

 These capabilities are very commonly needed, so this mixin is by default
 included into all `Ember.Component` instances.

 @class ContextBoundEventListenersMixin
 @public
 */
export default Mixin.create({
  init() {
    this._super(...arguments);

    this._listeners = [];
  },


  /**
   Attaches an event listener that will automatically be removed when the host
   object is dropped from DOM.

   Example:

   ```js
   import Component from 'ember-component';
   import ContextBoundEventListenersMixin from 'web-client/mixins/context-bound-event-listeners';

   export default Component.extend(ContextBoundEventListenersMixin, {
     didInsertElement() {
       this.addEventListener('.some-item', 'click', (e) => {
         console.log('.some-item was clicked');
       });
     }
   });
   ```

   @method addEventListener
   @param { String } selector the jQuery selector or element
   @param { String } eventName the event name to listen for
   @param { Function } callback the callback to run for that event
   */
  addEventListener(selector, _eventName, _callback) {
    assert(`Must provide an element (not a jQuery selector) when using addEventListener in a tagless component.`, this.tagName !== '' || typeof selector !== 'string');
    assert(`Called addEventListener before the component was rendered`, this._currentState === this._states.inDOM);

    let element = findElement(this.element, selector);
    let eventNames = buildEventNames(this, _eventName);
    let callback = run.bind(this, _callback);
    for (let i = 0; i < eventNames.length; i++) {
      let eventName = eventNames[i];
      element.addEventListener(eventName, callback);
      this._listeners.push({element, eventName, callback });
    }
  },

  willDestroyElement() {
    this._super();
    for (let i=0;i<this._listeners.length;i++) {
      let {element, eventName, callback} = this._listeners[i];
      element.removeEventListener(eventName, callback);
    }
    this._listeners.length = 0;
  }

});

function findElement(contextElement, selector) {
  let selectorType = typeof selector;
  let element;
  if (selectorType === 'string') {
    element = contextElement.querySelector(selector);
  } else if (selector.nodeType || selector === window) {
    element = selector;
  }

  assert(`Called addEventListener with bad selector value ${selector}`, !!element);

  return element;
}

function buildEventNames(element, _eventName) {
  let _eventNames = _eventName.split(' ');
  let eventNames = [];
  for (let i=0;i<_eventNames.length;i++) {
    assert(`Called addEventListener with an eventName containing a ., which is disallowed`, _eventName.indexOf('.') === -1);
    eventNames.push(_eventNames[i]);
  }
  return eventNames;
}
