import Service from 'ember-service';
import injectService from 'ember-service/inject';
import RSVP from 'rsvp';

const CORS_OPTIONS = {
  method: 'GET',
  mode: 'cors',
  cache: 'force-cache'
};

export default Service.extend({
  ajax: injectService(),

  host: 'badges.twitch.tv',

  init() {
    this._super(...arguments);

    this.set('_channels', {});
    this.set('collection', {
      global: {},
      channels: {}
    });

    this.loadGlobalBadges();
  },

  loadGlobalBadges() {
    let globalPromise = this.get('_globalPromise');
    if (globalPromise) {
      return globalPromise;
    }

    let globalUrl = `https://${this.get('host')}/v1/badges/global/display?language=en`;
    globalPromise = this.get('ajax').request(globalUrl, CORS_OPTIONS).then((result) => {
      if (this.isDestroying || this.isDestroyed) { return; }

      let badgeSet = result.badge_sets || {};
      this.set('collection.global', badgeSet);
    });
    this.set('_globalPromise', globalPromise);
    return globalPromise;
  },

  loadChannelBadges(channelId) {
    let channelKey = `_channels.${channelId}`;
    let channelPromise = this.get(channelKey);
    if (channelPromise) {
      return channelPromise;
    }

    let channelUrl = `https://${this.get('host')}/v1/badges/channels/${channelId}/display?language=en`;
    channelPromise = this.get('ajax').request(channelUrl, CORS_OPTIONS).then((result) => {
      if (this.isDestroying || this.isDestroyed) { return; }

      let badgeSet = result.badge_sets || {};
      this.set(`collection.channels.${channelId}`, badgeSet);
    });
    this.set(channelKey, channelPromise);
    return channelPromise;
  },

  loadBadges(channelId) {
    return RSVP.all([
      this.loadGlobalBadges(),
      this.loadChannelBadges(channelId)
    ]);
  },

  parseBadges(channelId, badgesTag) {
    if (!badgesTag) { return []; }

    let globalBadges = this.get('collection.global');
    let channelBadges = this.get(`collection.channels.${channelId}`);
    let badges = [];

    for (let badgeType in badgesTag) {
      let badgeVersion = badgesTag[badgeType];

      let matchedBadge = channelBadges[badgeType] || globalBadges[badgeType];
      if (matchedBadge) {
        let badgeData = matchedBadge.versions[badgeVersion];
        let srcSet = '';
        if (badgeData.image_url_2x) {
          srcSet = `${badgeData.image_url_2x} 2x`;
          if (badgeData.image_url_4x) {
            srcSet += `, ${badgeData.image_url_4x} 4x`;
          }
        } else if (badgeData.image_url_4x) {
          srcSet = `${badgeData.image_url_4x} 4x`;
        }

        badges.push({
          src: badgeData.image_url_1x,
          srcSet: srcSet,
          title: badgeData.title
        });
      }
    }

    return badges;
  }
});
