var querystring = require('querystring')
var validateSlackRequest = require('./validateSlackRequest')
var SlackPayload = require('./prototypes/SlackPayload')
var configureSlackResponder = require('./configureSlackResponder')

/**
 * Handler
 *
 * Knows what Slack is and how it sends requests.
 */
async function PrimeToolsLambda(event, context, callback) {

  console.log('Received event')
  console.log(event)

  // Validate that the incoming request is from Twitch Slack.
  try {
    validateSlackRequest({
      body: event.body,
      signature: event.headers['X-Slack-Signature'],
      timestamp: event.headers['X-Slack-Request-Timestamp'],
    })
  } catch (e) {
    let error = 'Invalid request: ' + e.message
    console.error(error)
    return {
      statusCode: 403,
      body: error
    }
  }

  console.log('Acknowledging request.')
  callback(null, process.env.NODE_ENV === 'production' ?
    { statusCode: 200, body: JSON.stringify({ response_type:  'in_channel' })}
  : { statusCode: 200 }
  )

  // Deserialize Slack payload and set up responder function.
  const { command, text, responseUrl, channelId, userId, userName } = new SlackPayload(querystring.parse(event.body))
  const sendResponse = configureSlackResponder(responseUrl)

  console.log('Echoing command.')
  await sendResponse(`>${ userName }: ${ command } ${ text }`)

  const whitelistedChannels = [
    'GC5GLD5FB', // samtools
    'G8WR7GFRB' // prime-turboest
  ]

  const whitelistedUsers = [
    'U6YL195AQ' // sunyang713
  ]

  if (!whitelistedChannels.includes(channelId) && !whitelistedUsers.includes(userId)) {
    let msg = 'Unauthorized channel.'
    console.log(msg)
    return sendResponse(msg)
  }

  // Extract arguments.
  const args = text.split(' ')

  // Match command.
  try {
    var cmd = require(`./commands/${ args[0] }`)
  } catch (e) {
    let msg = `Unrecognized command: ${ args[0] }. Run \\sam man to see a list of available commands.`
    console.log(msg)
    return sendResponse(msg)
  }

  // Execute command.
  try {
    var output = await cmd(...args.slice(1))
  } catch (e) {
    let error = `Command failed: ${ e.message }`
    console.error(error)
    return sendResponse(error)
  }

  console.log('Command finished')
  console.log(output)
  return sendResponse(output)
}

module.exports = PrimeToolsLambda
