"use strict";
var _this = this;
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var child_process_1 = require("child_process");
var chokidar = require("chokidar");
var crypto = require("crypto");
var fs = require("fs");
var lodash_1 = require("lodash");
var path = require("path");
var utils = require("../../../utils");
// tslint:disable-next-line
var currentCorePackage = require('carbon-components-prototype/package.json');
var coreRepoURI = currentCorePackage.repository.url;
// Set this to '0.0.0' when in test mode so that snapshots don't thrash on each version bump.
var coreVersion = process.env.NODE_ENV === 'test' ? '0.0.0' : currentCorePackage.version;
var typedocFilePath = path.join(__dirname, '/dist/___typedoc.json');
/**
 * Create Gatsby nodes for the documentation of each component – watches for
 * changes to component files, recompiles TypeDoc and resources nodes on change.
 */
exports.sourceNodes = function (_a, // tslint:disable-line no-any
_b) {
    var boundActionCreators = _a.boundActionCreators, reporter = _a.reporter;
    var componentSrc = _b.componentSrc;
    return tslib_1.__awaiter(_this, void 0, void 0, function () {
        var createNode, watcher, createAndProcessNodes;
        return tslib_1.__generator(this, function (_c) {
            createNode = boundActionCreators.createNode;
            watcher = chokidar.watch(componentSrc, {
                ignored: [
                    "**/*.scss",
                    "**/*.test.tsx",
                ],
            });
            createAndProcessNodes = function () {
                createTypedocFile(componentSrc);
                var fileContents = readTypedocFile();
                var components = filterOnlyComponents(fileContents);
                // Loop through the components.
                components.forEach(function (component, index) {
                    if (component.children) {
                        var info = component.children.find(function (child) { return (child.kindString !== undefined && !!child.kindString.match(/(Function|Class)/)); });
                        if (info) {
                            createNode({
                                id: "TypeDoc  " + utils.getComponentIdFromPath(component.originalName),
                                component: utils.getComponentIdFromPath(component.originalName),
                                name: info.name,
                                comment: info.comment,
                                properties: getProperties(component, fileContents),
                                parent: null,
                                children: [],
                                internal: {
                                    type: 'TypeDocFile',
                                    contentDigest: crypto
                                        .createHash('md5')
                                        .update(JSON.stringify(fileContents))
                                        .digest('hex'),
                                },
                            });
                        }
                        if (fileContents.children && fileContents.children.length === index + 1) {
                            reporter.success('fetch component data');
                        }
                    }
                });
            };
            createAndProcessNodes();
            watcher.on('change', lodash_1.debounce(function () {
                reporter.info('component file changed – rebuilding typedoc file');
                createAndProcessNodes();
            }, 500));
            return [2 /*return*/];
        });
    });
};
function getProperties(component, fileContents) {
    var properties = [];
    if (typeof component.children === 'undefined') {
        return properties;
    }
    component.children
        .filter(function (child) {
        if (!child.name) {
            return false;
        }
        if (!child.children) {
            return false;
        }
        return child.name.match(/Props$/) && !child.name.match(/BreakpointProps$/);
    })
        .map(function (child) {
        if (!child.children) {
            return;
        }
        child.children.map(function (nextChild) {
            // Check for duplicates
            if (properties.find(function (prop) { return prop.name === nextChild.name; })) {
                return;
            }
            var availableValues = [];
            var firstLevelValues = getFirstLevelValues(nextChild, fileContents);
            var secondLevelValues = getSecondLevelValues(nextChild, fileContents);
            if (firstLevelValues) {
                availableValues = availableValues.concat(firstLevelValues);
            }
            if (secondLevelValues) {
                availableValues = availableValues.concat(secondLevelValues);
            }
            // TODO: Consolidate properties.availableValues and properties.type and properties.sourceSrc
            var sourceRef = findTypedocItem(nextChild.type.id, fileContents);
            // Push the property.
            properties.push({
                name: nextChild.name,
                availableValues: availableValues,
                flags: nextChild.flags,
                comment: nextChild.comment,
                breakpointCompatible: !!(nextChild.inheritedFrom && nextChild.inheritedFrom.name.match(/BreakpointProps/)),
                type: getType(nextChild),
                sourceSrc: sourceRef && getSrcToSource(sourceRef.sources, coreRepoURI, coreVersion),
            });
        });
    });
    return properties;
}
function getFirstLevelValues(child, fileContents) {
    var id = child.type.id;
    var typeIsArray = false;
    // If it's an array of another type
    if (!id && child.type.type === 'array') {
        id = child.type.elementType.id;
        typeIsArray = true;
    }
    var ref = findTypedocItem(id, fileContents);
    var value;
    if (ref && ref.children) {
        value = buildValue(ref.id, fileContents);
    }
    else if (ref && ref.type && ref.type.types && ref.type.types.filter(function (i) { return i.type === 'unknown'; }).length > 0) {
        value = buildValue(ref.id, fileContents);
    }
    // If it's an array we'll just re-name the type to be clear
    if (value && typeIsArray) {
        value.name = value.name + '[]';
    }
    if (value) {
        return [value];
    }
    return [];
}
function getSecondLevelValues(child, fileContents) {
    var ref = findTypedocItem(child.type.id, fileContents);
    var nestedTypesWithIDS = child.type.types && child.type.types.filter(function (t) { return t.id !== undefined; });
    var returnValue = [];
    if (ref && ref.type && ref.type.types) {
        ref.type.types.forEach(function (item) {
            if (item.type === 'unknown') {
                return;
            }
            var value = buildValue(item.id, fileContents);
            if (value) {
                returnValue.push(value);
            }
        });
    }
    else if (nestedTypesWithIDS) {
        nestedTypesWithIDS.forEach(function (data) {
            if (data.id === undefined) {
                return;
            }
            var value = buildValue(data.id, fileContents);
            if (value) {
                returnValue.push(value);
            }
        });
    }
    return returnValue;
}
function buildValue(id, fileContents) {
    var ref = findTypedocItem(id, fileContents);
    if (!ref) {
        return;
    }
    // Handles enumerations
    if (ref.kindString === 'Enumeration') {
        return {
            name: ref.name,
            sourceSrc: getSrcToSource(ref.sources, coreRepoURI, coreVersion),
            values: ref.children.map(function (item) { return item.name; }),
        };
    }
    // Handles Enumeration Members
    if (ref.kindString === 'Enumeration member') {
        var refParent = findTypedocParent(id, fileContents);
        return {
            name: refParent && refParent.name,
            sourceSrc: getSrcToSource(ref.sources, coreRepoURI, coreVersion),
            values: [ref.name],
        };
    }
    // Handles nested types
    if (ref.type && ref.type.types) {
        return {
            name: ref.name,
            sourceSrc: getSrcToSource(ref.sources, coreRepoURI, coreVersion),
            values: ref.type.types.map(function (item) { return (item.name || item.value); }),
        };
    }
    // Handles nested children
    if (ref.children) {
        return {
            name: ref.name,
            sourceSrc: getSrcToSource(ref.sources, coreRepoURI, coreVersion),
            properties: ref.children.map(function (item) {
                // This may be a reference to another defined type or interface somewhere, or an array of a defined type!
                var refId = (item.type && item.type.id) || (item.type && item.type.type === 'array' && item.type.elementType && item.type.elementType.id);
                var itemRef = findTypedocItem(refId, fileContents);
                var sourceSrc = itemRef && itemRef.sources && getSrcToSource(itemRef.sources, coreRepoURI, coreVersion);
                return {
                    name: item.name,
                    value: getType(item),
                    isOptional: !!item.flags.isOptional,
                    sourceSrc: sourceSrc,
                };
            }),
        };
    }
}
function getType(child) {
    var type;
    var typeIsArray = false;
    // If it's an array of another type
    if (child.type.type === 'array') {
        typeIsArray = true;
    }
    var referenceData = child.type.types && child.type.types.filter(function (t) { return t.type === 'reference'; });
    if (referenceData && referenceData.length > 0) {
        type = referenceData.reduce(function (accumulator, item, index) {
            if (item.name !== undefined) {
                return index > 0 ? accumulator + " | " + item.name : item.name;
            }
            else {
                return accumulator;
            }
        }, '');
    }
    if (child.type.types && child.type.types.find(function (t) { return t.name === 'string'; })) {
        type = 'string';
    }
    else if (child.type.types && child.type.types.find(function (t) { return t.name === 'number'; })) {
        type = 'number';
    }
    else if (child.type.types && child.type.types.find(function (t) { return t.name && t.name.match(/(true|false)/); })) {
        type = 'boolean';
    }
    else if (child.type.name) {
        type = child.type.name;
    }
    else if (child.type.elementType && child.type.elementType.name) {
        type = child.type.elementType.name;
    }
    if (typeIsArray) {
        type = type + '[]';
    }
    return type;
}
/**
 * Returns the node which exactly matches the ID provided
 *
 * @param id - Node ID to search by
 * @param data TypeDoc file data node
 */
function findTypedocItem(id, data) {
    if (data.id === id) {
        return data;
    }
    if (!Array.isArray(data.children)) {
        return undefined;
    }
    for (var i = 0; i < data.children.length; i++) {
        var found = findTypedocItem(id, data.children[i]);
        if (found !== undefined) {
            return found;
        }
    }
}
/**
 * Returns the parent node which contains ID provided as one of it's children
 *
 * @param id - Child node ID to search by
 * @param data TypeDoc file data node
 */
function findTypedocParent(id, data) {
    if (!Array.isArray(data.children)) {
        return undefined;
    }
    if (data.children.findIndex(function (child) { return child.id === id; }) > -1) {
        return data;
    }
    for (var i = 0; i < data.children.length; i++) {
        var found = findTypedocParent(id, data.children[i]);
        if (found !== undefined) {
            return found;
        }
    }
    return undefined;
}
function getSrcToSource(sources, repoUrl, tagVersion) {
    if (sources.length === 0) {
        return;
    }
    if (!sources[0].fileName) {
        return;
    }
    if (!sources[0].line) {
        return;
    }
    if (!repoUrl) {
        return;
    }
    var fileName = sources[0].fileName;
    var line = sources[0].line;
    var urlBase = repoUrl.replace(':', '/').replace('.git', '/').replace('git@', 'https://'); // Expects format: git@git-aws.internal.justin.tv:core-ui/core-ui.git
    var branch = tagVersion ? "blob/v" + tagVersion + "/" : 'blob/master/';
    var dir = "src/";
    return urlBase + branch + dir + fileName + '#L' + line;
}
function createTypedocFile(sourceDir) {
    // Note: Order of arguments is important.
    child_process_1.execSync("typedoc --json " + typedocFilePath + " " + sourceDir + " --exclude \"**/{*{index.ts,test.tsx},/tests/*}\" --excludeExternals --ignoreCompilerErrors ");
}
function readTypedocFile() {
    if (fs.existsSync(typedocFilePath)) {
        var fileContents = JSON.parse(fs.readFileSync(typedocFilePath, 'utf8'));
        return fileContents;
    }
}
function filterOnlyComponents(fileContents) {
    if (!fileContents || typeof fileContents.children === 'undefined') {
        return [];
    }
    return fileContents.children.filter(function (item) {
        return (item.name.search('"components/') === 0);
    });
}
module.exports.getSrcToSource = getSrcToSource;
module.exports.getProperties = getProperties;
module.exports.createTypedocFile = createTypedocFile;
module.exports.readTypedocFile = readTypedocFile;
module.exports.filterOnlyComponents = filterOnlyComponents;
/**
 * Resources:
 *  - https://github.com/gatsbyjs/gatsby/blob/master/packages/gatsby-source-filesystem/src/gatsby-node.js
 */
