---
description: Custom – or feature – styles may be necessary to build new features on Twitch platforms; we also have tools and guidlines to help build these features in a consistent and maintainable way.
---

Use variables for the following delcarations:
* `font-size`
* `font-weight`
* `font-family`
* `margin`
* `padding`
* `z-index`

## Typography

### Font Sizes

| Name | Default Value |
|------|---------------|
| `$font-size-1` | `4rem` |
| `$font-size-2` | `2.8rem` |
| `$font-size-3` | `2.4rem` |
| `$font-size-4` | `1.6rem` |
| `$font-size-5` | `1.4rem` |
| `$font-size-6` | `1.2rem` |


### Font Weight

| Name | Default Value |
|------|---------------|
| `$font-weight-bold` | `600` |
| `$font-weight-normal` | `400` |


### Font Families

| Name | Default Value |
|------|---------------|
| `$font-base` | `'Helvetica Neue', Helvetica, Arial, sans-serif`|


### Line Heights

| Name | Default Value |
|------|---------------|
| `$line-height-heading` | `1.2` |
| `$line-height-body` | `1.5` |


## Spacing
Use these variables on `margin` and `padding` declarations to ensure consistent spacing across the platform.

| Name | Default Value |
|------|---------------|
| `$space-0` | `0` |
| `$space-05` | `0.5rem` |
| `$space-1` | `1rem` |
| `$space-2` | `2rem` |
| `$space-3` | `3rem` |
| `$space-4` | `4rem` |
| `$space-5` | `5rem` |


## Positioning

### Z-Index

`z-index` is an property that can become very difficult to manage. To help make it easier to manage, we've separated z-index into two "scopes".

#### Relative
Relative indices should be used to describe elements that are positioned within a component.

| Name | Default Value |
|------|---------------|
| `$z-index-above` | `10` |
| `$z-index-default` | `1` |
| `$z-index-below` | `-1` |

#### Absolute
Absolute indinces should only be used for higher level elements on a page. These are elements that are positioned relatively to the viewport, as opposed to a single component.

| Name | Default Value |
|------|---------------|
| `$z-index-modal` | `5000` |
| `$z-index-notification` | `4000` |
| `$z-index-overlay` | `3000` |
| `$z-index-balloon` | `2000` |
| `$z-index-sticky` | `1000` |


## Units
For all properties that are not stylistic, we utilize `rem` units to allow for proper scaling between devices and resolution settings. `rem`'s are based on
the font-size of the body. At Twitch, we have a base of `10px` so that `1rem` = `10px`.


## Naming Selectors
We utilize the BEM naming structure within our component styles to allow CSS to be used outside of React environments. The BEM methodology couples well with component-based thinking in that simpler, reusable components should be favored over a complex block. Using BEM selectors eliminates issues caused by inheritance that occurs when selecting elements through specificity.

_Tip: If you find yourself needing to write a second element, odds are that you should consider a new component all together._

```scss no-copy
.block {

  &__element {
    &--modifier { }
  }

  &--modifier { }
}
```

## Scope
Component styles should only select elements that are in their corresponding component mark up. Selecting elements outside of this scope will not work in React and leads to inheritance issues that are hard to track outside of React environments.

## Media Queries
In an effort to adopt mobile-first mindset, our media queries add styles progressively through `min-width`. Styles outside of media queries are always applied, and styles within media queries are applied from the width of the breakpoint and up.
Breakpoints should only be accessed through the `mq` mixin that excepts the following keys: `xs`, `sm`, `md`, `lg`, `xl`, and `xxl`.

```scss
@include mq('sm');
```

### Breakpoint Reference

| Key  | Min. Screen Width |
|------|---------------|
| `xs` | `480px` |
| `sm` | `768px` |
| `md` | `1024px` |
| `lg` | `1200px` |
| `xl` | `1440px` |
| `xxl` | `1600px` |


## Comments

Use SCSS style comments (`//`) in your files to help logically break up files
and to document what the styles do. When writing block and section heading
comments, ensure the lines of hyphens end on column `80`.

### Block Headings

Use block heading comments to add information about block selectors near the
top of files.

```scss no-copy
// ----------------------------------------------------------------------------
// Block Heading
// ----------------------------------------------------------------------------
// Staghorn sculpin char sea catfish nibbler false trevally, rockfish.
// Bocaccio prickleback, anglerfish Billfish nurseryfish lanternfish hog
// sucker: tiger shark silver carp surfperch.
```

### Section Headings

Use a section heading to help group selectors that are related, such as a list
of modifiers or elements within a block.

```scss no-copy
// Section Heading
// ----------------------------------------------------------------------------
```


## Linting Rules

### Single Quotes
Use single-quotes for everything.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss no-copy
    .selector {
      background-image: url('path/to/image.png');

      &::before {
        content: '';
      }

      &[type='text'] {}
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    .selector {
      background-image: url("path/to/image.png");

      &::before {
        content: "";
      }

      &[type="text"] {}
    }
    ```
```

### BEM Depth
Blocks should only have one level of elements.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .block {
      &__element { }
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    .block {
      &__element {
        &__element { }
      }
    }
    ```
```

### Declarations Before Nesting
Declarations within a selector should be written before nested selectors.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .block {
      display: block;

      &__element { }
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    .block {
      &__element { }

      display: block;
    }
    ```
```

### Empty Arguments
Write mixins without parentheses if no arguments are being supplied.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .block {
      @include theme {
        background: token('color-background');
      }
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    .block {
      @include theme() {
        background: token('color-background');
      }
    }
    ```
```


### Empty Line Between Blocks
Leave a line between the declarations of the previous selector.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .block {
      display: block;
    }

    .another-block {
      display: block;
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    .block {
      display: block;
    }
    .another-block {
      display: block;
    }
    ```
```

### Mixins before Declarations
Call mixins at the top of each selector before other declarations. This concept is similar to importing files at the top of a file in React.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    @include theme {
      color: token('color-text-base');
    }

    display: block;
    ```
-
  type: 'dont'
  body: |
    ```scss
    display: block;

    @include theme {
      color: token('color-text-base');
    }
    ```
```

### Force Nesting

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .block {
      &::before {}
      &:hover {}
      &[type='text'] {}
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    display: block;

    @include theme {
      color: token('color-text-base');
    }
    ```
```

### Function/Mixin Name Format
Functions and mixin names should be all lowercase and utilize hyphens to logically separate words.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    @function function-name($args) {};
    ```
-
  type: 'dont'
  body: |
    ```scss
    @function functionName($args) {};
    @function function_name($args) {};
    ```
```

### Leading Zero

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    top: 0.5rem
    ```
-
  type: 'dont'
  body: |
    ```scss
    top: .5rem
    ```
```

### Color Keywords/Literals

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    @include theme {
      color: token('color-text-base');
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    color: #000
    color: rgb(255, 255, 255)
    color: black
    ```
```

### Nesting Depth
Nesting depth should not exceed 3 levels. If you find yourself needing to nest selectors past 3 levels, take another look at your markup and component structure – you may need to break up larger components into smaller ones.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .block {
      &__depth-2 {
        &--depth-3 { }
      }
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    .block {
      &__depth-2 {
        &--depth-3 {
          &--depth-4 {
            &--depth-5 { }
          }
        }
      }
    }
    ```
```

### Combinators

Use good classnames to select elements. Combinators can be difficult to follow and can break easily if tags or other classnames change.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .card {
      &__title {}
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    .card {
      & > h1 {}
      & > .title {}
      & + div {}
    }
    ```
```

### CSS Comments
CSS comments are not allowed, use SCSS comments instead.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    // This is a SCSS comment.
    ```
-
  type: 'dont'
  body: |
    ```scss
    /* This is a CSS Comment */
    ```
```

### Duplicate Properties

```yml block-guidelines
-
  type: 'dont'
  body: |
    ```scss
    .block {
      display: block;
      position: relative;
      display: inline-block; // Duplicated property.
    }
    ```
```

### Empty Rulesets
All selectors must have declarations.

### Extends
Extends are not allowed.

```yml block-guidelines
-
  type: 'dont'
  body: |
    ```scss
    @extend .some-class;
    ```
```

### ID Selectors


```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .class-name-selector { }
    ```
-
  type: 'dont'
  body: |
    ```scss
    #id-selector {}
    ```
```

### Important
The `!important` tag is not allowed.

```yml block-guidelines
-
  type: 'dont'
  body: |
    ```scss
    .block {
      display: block !important;
    }
    ```
```

### Mergeable Selectors

```yml block-guidelines
-
  type: 'dont'
  body: |
    ```scss
    .selector {
      position: absolute;
    }

    .selector {
      top: 0
      left: 0
    }
    ```
```

### Transition All
The use of `transition: all` is not allowed. Be sure to specify only the properties that are being transitioned.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .block {
      transition: background-color 0.2s ease-in;
    }
    ```
```

### Universal Selectors
The use of the `*` selector is not allowed.

```yml block-guidelines
-
  type: 'dont'
  body: |
    ```scss
    * {
      display: block;
    }
    ```
```

### Vendor Prefixes
We use `autoprefixer` as part of our build process, so vendor prefix tags (`-moz`, `-webkit`, etc.) are not needed.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .block {
      transform: scale(0.5);
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    .block {
      -webkit-transform: scale(0.5);
          -moz-transform: scale(0.5);
          -ms-transform: scale(0.5);
              transform: scale(0.5);
    }
    ```
```

### Pseudo Element Selectors

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .selector {
      &::before {}
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    .selector {
      &:before {}
    }
    ```
```

### Single Line Per Selector
Each selector should appear on it's own line – this makes them easier to read and easier to delete.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .selector,
    .another {
      display: block;
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    .selector, .another {
      display: block;
    }
    ```
```

### Shorthand

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .selector {
      padding: $space-1 $space-2;
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    .selector {
      padding-top: $space-1;
      padding-right: $space-2;
      padding-bottom: $space-1;
      padding-left: $space-1;
    }
    ```
```

### Spaces

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .selector {
      display: block;
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    .selector{
      display:block;
    }
    ```
```

### Variable Name Format
Variable names should be lowercase and use hyphens to separate words.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    $variable-name-format: 1rem;
    ```
```

### Zero Unit
If the value of a property is 0, omit the unit.

```yml block-guidelines
-
  type: 'do'
  body: |
    ```scss
    .selector {
      margin: 0;
    }
    ```
-
  type: 'dont'
  body: |
    ```scss
    .selector {
      margin: 0rem;
      margin: 0px;
    }
    ```
```
