---
component:
  id: 'components/layout'
  disablePlayground: true
---

Layout comes in many flavors: `<Layout>`, `<StyledLayout>`, `<InjectLayout>`, and `<InjectStyledLayout>`. You should reach for them in that order, with preference for the simplest component which solves the problem.

These components provide a typed API to apply CSS properties (such as padding, margin, or FlexBox properties) to wrapper such as a `<div>` tag. [See docs on Styled Layout](./styled-layout) for more info about visual layout styles, or [read up on CSS FlexBox](https://css-tricks.com/snippets/css/a-guide-to-flexbox/) to learn more about the available properties.

## Example: FlexBox with images and text

A common layout pattern which includes text, images, and CTA buttons within a rectangular area. Notice that the center region with the text is designated to grow and shrink as the available space resizes and the other two regions do not grow or shrink.

```jsx
<StyledLayout display={Display.Flex} elevation={1} background={Background.Base}>

  <Layout flexGrow={0} flexShrink={0}>
    <img src="https://static-cdn.jtvnw.net/ttv-boxart/StarCraft%20II-95x127.jpg" />
  </Layout>

  <Layout flexGrow={1} flexShrink={1} padding={2}>
    <p>Starcraft 2 is the most awesome game ever!</p>
  </Layout>

  <Layout flexGrow={0} flexShrink={0} padding={2}>
    <Button>Buy Now</Button>
  </Layout>

</StyledLayout>
```

## Example: Vertical centering elements in a row

By setting a flex-child to ALSO behave as a flex-parent, we can vertically center things:

```jsx
<StyledLayout display={Display.Flex} border padding={1}>
  <Button>Click Me</Button>
  <Layout display={Display.Flex} alignItems={AlignItems.Center} margin={{ left: 1 }}>
    <p>This text is vertically centered!</p>
  </Layout>
</StyledLayout>
```

## Example: Stacking and anchoring

```jsx
<Layout position={Position.Relative}>
  <img src="https://s3-us-west-2.amazonaws.com/web-design-int-production/assets/brand/images/Photography--4.jpg" width="100%" />
  <StyledLayout
    position={Position.Absolute}
    attachTop
    attachRight
    padding={2}
    margin={2}
    background={Background.Base}
    elevation={3}
  >
    <CoreText bold>An element positioned in the top right.</CoreText>
  </StyledLayout>
</Layout>
```

## Example: Stacking and centering

```jsx
<Layout position={Position.Relative}>
  <img src="https://s3-us-west-2.amazonaws.com/web-design-int-production/assets/brand/images/Photography--4.jpg" width="100%" />
  <Layout
    position={Position.Absolute}
    attachTop
    attachLeft
    fullWidth
    fullHeight
    display={Display.Flex}
    alignItems={AlignItems.Center}
    justifyContent={JustifyContent.Center}
  >
    <StyledLayout
      padding={2}
      background={Background.Base}
      elevation={3}
    >
      <CoreText bold>An element positioned in the center.</CoreText>
    </StyledLayout>
  </Layout>
</Layout>
```

## Example: Flex Columns (All Viewport Sizes)

```jsx
<Layout
  display={Display.Flex}
  flexDirection={FlexDirection.Column}
>
  <Layout margin={0.5}><Stat label="My Cool Label" value="1234" icon={SVGAsset.NavDashboard} /></Layout>
  <Layout margin={0.5}><Stat label="My Cool Label" value="4567" icon={SVGAsset.DeadGlitch} /></Layout>
  <Layout margin={0.5}><Stat label="My Cool Label" value="8910" icon={SVGAsset.Bits} /></Layout>
</Layout>
```

## Example: Responsive Flex Columns and Rows
This will create **rows at large viewports** and **columns at small viewports**. You can re-size your browser viewport to see this example switch between Rows and Colummns.

```jsx
<Layout
  display={Display.Flex}
  flexDirection={FlexDirection.Column}
  breakpointMedium={{
    flexDirection: FlexDirection.Row
  }}
>
  <Layout margin={0.5}><Stat label="My Cool Label" value="1234" icon={SVGAsset.NavDashboard} /></Layout>
  <Layout margin={0.5}><Stat label="My Cool Label" value="5678" icon={SVGAsset.DeadGlitch} /></Layout>
  <Layout margin={0.5}><Stat label="My Cool Label" value="8910" icon={SVGAsset.Bits} /></Layout>
</Layout>
```

## Padding

### All Sides
```jsx
<div>
<StyledLayout padding={1} margin={1} display={Display.InlineBlock} background={Background.Accent} />
<StyledLayout padding={2} margin={1} display={Display.InlineBlock} background={Background.Accent} />
<StyledLayout padding={3} margin={1} display={Display.InlineBlock} background={Background.Accent} />
<StyledLayout padding={4} margin={1} display={Display.InlineBlock} background={Background.Accent} />
<StyledLayout padding={5} margin={1} display={Display.InlineBlock} background={Background.Accent} />
</div>
```

### By Side
```jsx
<div>
<StyledLayout padding={{ x: 1, y: 2 }} margin={1} display={Display.InlineBlock} background={Background.Accent}>
  <CoreText color={Color.Overlay}>•</CoreText>
</StyledLayout>
<StyledLayout padding={{ x: 4, y: 2 }} margin={1} display={Display.InlineBlock} background={Background.Accent}>
  <CoreText color={Color.Overlay}>•</CoreText>
</StyledLayout>
<StyledLayout padding={{ left: 5, right: 3, y: 2 }} margin={1} display={Display.InlineBlock} background={Background.Accent}>
  <CoreText color={Color.Overlay}>•</CoreText>
</StyledLayout>
</div>
```
