const fs = require('fs');
const path = require('path');
const utils = require(path.join(__dirname, '..', 'utils.js'));
const handlebars = require('handlebars');
const argv = require('minimist')(process.argv.slice(2));

if (argv._.length === 0) {
  console.error(`🚫  ERROR - Incorrect number of arguments. See usage for examples.`);
  process.exit(utils.exitCodes.error);
}

// Name of the component
const inputComponentName = argv._[0];
// Name of the parent group (optional)
const parentGroup = argv.parentGroup ? argv.parentGroup : ( (argv.pg ? argv.pg : '') );
// Boolean helper if parent group exists
const hasParentGroup = (parentGroup !== '');
// Absolute path to the folder
const absoluteFolder = path.join(process.cwd(), 'src', 'components', parentGroup);
// Hyphenated component name.
const nameHyphen = utils.toHyphens(inputComponentName);
// Absolute path to component folder.
const absolutePath = path.join(absoluteFolder, nameHyphen);

console.log(`\n🔥  Starting <${inputComponentName}> component build...`);

// Validate common arguments.
console.log(`   Checking formatting...`);
if (!inputComponentName.substring(0, 1).match(/[A-Z]/)) {
  console.error(`🚫  ERROR - Component <name> must be PascalCase: ${inputComponentName}`);
  process.exit(utils.exitCodes.error);
}

console.log(`   Checking directories...`);

// Check to see if the parent group directory exists.
if (!fs.existsSync(absoluteFolder)) {
  fs.mkdirSync(absoluteFolder);
}

// Create the component directory if it doesn't exist.
if (!fs.existsSync(absolutePath)) {
  fs.mkdirSync(absolutePath);
} else {
  console.error(`🚫  ERROR - a component with the name '${inputComponentName}' already exists in that directory.`);
  process.exit(utils.exitCodes.error);
}

// Create the component template file.
fs.writeFileSync(`${absolutePath}${path.sep}component.tsx`,
  handlebars.compile(
    fs.readFileSync(path.join(__dirname, 'component.template'), 'utf8')
  )({
    name: inputComponentName,
    nameHyphen,
    nameLower: utils.toFirstLower(inputComponentName),
    nameSpace: utils.toSpaces(inputComponentName),
    hasParentGroup,
  })
);

console.log(`   Creating ${absolutePath}${path.sep}component.tsx ...`);

// Create the styles template.
fs.writeFileSync(`${absolutePath}${path.sep}styles.sass`,
  handlebars.compile(
    fs.readFileSync(path.join(__dirname, 'styles.template'), 'utf8')
  )({nameHyphen})
);

console.log(`   Creating ${absolutePath}${path.sep}styles.sass ...`);

// Create the index template.
fs.writeFileSync(`${absolutePath}${path.sep}index.ts`,
  handlebars.compile(
    fs.readFileSync(path.join(__dirname, 'index.template'), 'utf8')
  )()
);

console.log(`   Creating ${absolutePath}${path.sep}index.ts ...`);

console.log(`✅  <${inputComponentName}> created successfully!`);
process.exit(utils.exitCodes.success);
