const { fork } = require('child_process');
const path = require('path');
const argv = require('minimist')(process.argv.slice(2));
const utils = require('./utils.js');

// Get arguments from input command, slicing the process.execPath and this file path.
const args = process.argv.slice(2);

if (argv._.length === 0) {
  usage();
  return;
}

switch (argv._[0]) {
  case 'component':
    const processArgs = args.slice(1);
    const inputOptions = argv._.slice(1);
    const inputComponentName = argv._[1];
    const componentGroupDirectory = argv.parentGroup ? `/${argv.parentGroup}` : ( (argv.pg ? `/${argv.pg}` : '') )
    const nameHyphen = utils.toHyphens(argv._[1]);
    const componentGenerator = fork(path.join(__dirname, 'component', 'component.js'), processArgs);

    componentGenerator.on('exit', (code) => {
      if (code === 1) { return; }
      const playgroundGenerator = fork(path.join(__dirname, 'playground', 'playground.js'), processArgs);

      playgroundGenerator.on('exit', (code) => {
        if (code === 1) { return; }
        console.log('\x1b[36m%s\x1b[0m', `
Once the component is ready for public API usage:
  [1] Export it from the src/index.ts
    export { ${inputComponentName} } from './components${componentGroupDirectory}/${nameHyphen}';

To add this component to the docs:
  [1] Add its page component to src/playground/main.tsx
    import { ${inputComponentName}Page } from './pages/components${componentGroupDirectory}/${nameHyphen}';
    ...
    <Route path="/user-interface/components${componentGroupDirectory}/${nameHyphen}" component={${inputComponentName}Page} />
  [2] Add its route to src/playground/components/main-nav/component.tsx
    { name: '${inputComponentName}', linkTo: '/user-interface/components${componentGroupDirectory}/${nameHyphen}' },

        `)
      });
    });

    return;
  default:
    usage();
    return;
}

function usage() {
  console.log(`
Usage: yarn gen -- component <ComponentName> [...]

    Create a Core UI component. Builds files component and Playground files.

    Blueprint:
      <ComponentName>
        The name of the component in PascalCase.

      [--parentGroup=group-name | --pg group-name]
        The name of the group that the core-ui component resides in.

    Examples:
      yarn gen -- component BestFeature
        src/components/best-feature
          component.tsx
          index.ts
          styles.sass
        src/playground/pages/components/best-feature
          component.tsx
          index.ts
          styles.sass

      yarn gen -- component AnotherFeature --pg graph
        src/components/graph/another-feature
          component.tsx
          index.ts
          styles.sass
        src/playground/pages/components/graph/another-feature
          component.tsx
          index.ts
          styles.sass
`);
}
