import { CoreLink, Layout } from 'carbon-components-prototype';
import * as React from 'react';

type AriaDefinitionList = {
  [key: string]: JSX.Element;
};

const WIA_ARIA_PATH = 'https://www.w3.org/TR/wai-aria-1.1/';

const AriaDefinitions: AriaDefinitionList = {
  'aria-activedescendant': (
    <dd>Identifies the currently active element when <abbr title="Document Object Model">DOM</abbr> focus is on a <a href="https://www.w3.org/TR/wai-aria-1.1/#composite" className="role-reference"><code>composite</code></a> widget, <a href="https://www.w3.org/TR/wai-aria-1.1/#textbox" className="role-reference"><code>textbox</code></a>, <a href="https://www.w3.org/TR/wai-aria-1.1/#group" className="role-reference"><code>group</code></a>, or <a href="https://www.w3.org/TR/wai-aria-1.1/#application" className="role-reference"><code>application</code></a>.</dd>
  ),
  'aria-atomic': (
    <dd>Indicates whether <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-assistive-technology" className="internalDFN" data-link-type="dfn">assistive technologies</a> will present all, or only parts of, the changed region based on the change notifications defined by the <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-relevant" className="property-reference"><code>aria-relevant</code></a> attribute.</dd>
  ),
  'aria-autocomplete': (
    <dd>Indicates whether inputting text could trigger display of one or more predictions of the user's intended value for an input and specifies how predictions would be presented if they are made.</dd>
  ),
  'aria-busy': (
    <dd>Indicates an element is being modified and that assistive technologies <em className="rfc2119" title="MAY">MAY</em> want to wait until the modifications are complete before exposing them to the user.</dd>
  ),
  'aria-checked': (
    <dd>Indicates the current "checked" <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-state" className="internalDFN" data-link-type="dfn">state</a> of checkboxes, radio buttons, and other <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-widget" className="internalDFN" data-link-type="dfn">widgets</a>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-pressed" className="state-reference"><code>aria-pressed</code></a> and <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-selected" className="state-reference"><code>aria-selected</code></a>.</dd>
  ),
  'aria-colcount': (
    <dd>Defines the total number of columns in a <a href="https://www.w3.org/TR/wai-aria-1.1/#table" className="role-reference"><code>table</code></a>, <a href="https://www.w3.org/TR/wai-aria-1.1/#grid" className="role-reference"><code>grid</code></a>, or <a href="https://www.w3.org/TR/wai-aria-1.1/#treegrid" className="role-reference"><code>treegrid</code></a>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-colindex" className="property-reference"><code>aria-colindex</code></a>.</dd>
  ),
  'aria-colindex': (
    <dd>Defines an <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element's</a> column index or position with respect to the total number of columns within a <a href="https://www.w3.org/TR/wai-aria-1.1/#table" className="role-reference"><code>table</code></a>, <a href="https://www.w3.org/TR/wai-aria-1.1/#grid" className="role-reference"><code>grid</code></a>, or <a href="https://www.w3.org/TR/wai-aria-1.1/#treegrid" className="role-reference"><code>treegrid</code></a>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-colcount" className="property-reference"><code>aria-colcount</code></a> and <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-colspan" className="property-reference"><code>aria-colspan</code></a>.</dd>
  ),
  'aria-colspan': (
    <dd>Defines the number of columns spanned by a cell or gridcell within a <a href="https://www.w3.org/TR/wai-aria-1.1/#table" className="role-reference"><code>table</code></a>, <a href="https://www.w3.org/TR/wai-aria-1.1/#grid" className="role-reference"><code>grid</code></a>, or <a href="https://www.w3.org/TR/wai-aria-1.1/#treegrid" className="role-reference"><code>treegrid</code></a>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-colindex" className="property-reference"><code>aria-colindex</code></a> and <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-rowspan" className="property-reference"><code>aria-rowspan</code></a>.</dd>
  ),
  'aria-controls': (
    <dd>Identifies the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> (or elements) whose contents or presence are controlled by the current element. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-owns" className="property-reference"><code>aria-owns</code></a>.</dd>
  ),
  'aria-current': (
    <dd>Indicates the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> that represents the current item within a container or set of related elements.</dd>
  ),
  'aria-describedby': (
    <dd>Identifies the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> (or elements) that describes the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-object" className="internalDFN" data-link-type="dfn">object</a>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-labelledby" className="property-reference"><code>aria-labelledby</code></a>.</dd>
  ),
  'aria-details': (
    <dd>Identifies the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> that provides a detailed, extended description for the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-object" className="internalDFN" data-link-type="dfn">object</a>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-describedby" className="property-reference"><code>aria-describedby</code></a>.</dd>
  ),
  'aria-disabled': (
    <dd>Indicates that the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> is <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-perceivable" className="internalDFN" data-link-type="dfn">perceivable</a> but disabled, so it is not editable or otherwise <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-operable" className="internalDFN" data-link-type="dfn">operable</a>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-hidden" className="state-reference"><code>aria-hidden</code></a> and <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-readonly" className="property-reference"><code>aria-readonly</code></a>.</dd>
  ),
  'aria-dropeffect': (
    <dd>[Deprecated in ARIA 1.1] Indicates what functions can be performed when a dragged object is released on the drop target.</dd>
  ),
  'aria-errormessage': (
    <dd>Identifies the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> that provides an error message for the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-object" className="internalDFN" data-link-type="dfn">object</a>.  See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-invalid" className="state-reference"><code>aria-invalid</code></a> and <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-describedby" className="property-reference"><code>aria-describedby</code></a>. </dd>
  ),
  'aria-expanded': (
    <dd>Indicates whether the element, or another grouping element it controls, is currently expanded or collapsed.</dd>
  ),
  'aria-flowto': (
    <dd>Identifies the next <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> (or elements) in an alternate reading order of content which, at the user's discretion, allows assistive technology to override the general default of reading in document source order.</dd>
  ),
  'aria-grabbed': (
    <dd>[Deprecated in ARIA 1.1] Indicates an element's "grabbed" <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-state" className="internalDFN" data-link-type="dfn">state</a> in a drag-and-drop operation.</dd>
  ),
  'aria-haspopup': (
    <dd>Indicates the availability and type of interactive popup element, such as menu or dialog, that can be triggered by an <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a>.</dd>
  ),
  'aria-hidden': (
    <dd>Indicates whether the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> is exposed to an accessibility <abbr title="Application Programing Interfaces">API</abbr>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-disabled" className="state-reference"><code>aria-disabled</code></a>.</dd>
  ),
  'aria-invalid': (
    <dd>Indicates the entered value does not conform to the format expected by the application. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-errormessage" className="property-reference"><code>aria-errormessage</code></a>.</dd>
  ),
  'aria-keyshortcuts': (
    <dd>Indicates keyboard shortcuts that an author has implemented to activate or give focus to an element.</dd>
  ),
  'aria-label': (
    <dd>Defines a string value that labels the current element. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-labelledby" className="property-reference"><code>aria-labelledby</code></a>.</dd>
  ),
  'aria-labelledby': (
    <dd>Identifies the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> (or elements) that labels the current element. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-describedby" className="property-reference"><code>aria-describedby</code></a>.</dd>
  ),
  'aria-level': (
    <dd>Defines the hierarchical level of an <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> within a structure.</dd>
  ),
  'aria-live': (
    <dd>Indicates that an <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> will be updated, and describes the types of updates the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-user-agent" className="internalDFN" data-link-type="dfn">user agents</a>, <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-assistive-technology" className="internalDFN" data-link-type="dfn">assistive technologies</a>, and user can expect from the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-live-region" className="internalDFN" data-link-type="dfn">live region</a>.</dd>
  ),
  'aria-modal': (
    <dd>Indicates whether an <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> is modal when displayed.</dd>
  ),
  'aria-multiline': (
    <dd>Indicates whether a text box accepts multiple lines of input or only a single line.</dd>
  ),
  'aria-multiselectable': (
    <dd>Indicates that the user may select more than one item from the current selectable descendants.</dd>
  ),
  'aria-orientation': (
    <dd>Indicates whether the element's orientation is horizontal, vertical, or unknown/ambiguous.</dd>
  ),
  'aria-owns': (
    <dd>Identifies an <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> (or elements) in order to define a visual, functional, or contextual parent/child <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-relationship" className="internalDFN" data-link-type="dfn">relationship</a> between <abbr title="Document Object Model">DOM</abbr> elements where the <abbr title="Document Object Model">DOM</abbr> hierarchy cannot be used to represent the relationship. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-controls" className="property-reference"><code>aria-controls</code></a>.</dd>
  ),
  'aria-placeholder': (
    <dd>Defines a short hint (a word or short phrase) intended to aid the user with data entry when the control has no value. A hint could be a sample value or a brief description of the expected format.</dd>
  ),
  'aria-posinset': (
    <dd>Defines an <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a>'s number or position in the current set of listitems or treeitems. Not required if all elements in the set are present in the <abbr title="Document Object Model">DOM</abbr>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-setsize" className="property-reference"><code>aria-setsize</code></a>.</dd>
  ),
  'aria-pressed': (
    <dd>Indicates the current "pressed" <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-state" className="internalDFN" data-link-type="dfn">state</a> of toggle buttons. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-checked" className="state-reference"><code>aria-checked</code></a> and <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-selected" className="state-reference"><code>aria-selected</code></a>.</dd>
  ),
  'aria-readonly': (
    <dd> Indicates that the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> is not editable, but is otherwise <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-operable" className="internalDFN" data-link-type="dfn">operable</a>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-disabled" className="state-reference"><code>aria-disabled</code></a>.</dd>
  ),
  'aria-relevant': (
    <dd>Indicates what notifications the user agent will trigger when the accessibility tree within a live region is modified. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-atomic" className="property-reference"><code>aria-atomic</code></a>.</dd>
  ),
  'aria-required': (
    <dd>Indicates that user input is required on the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a> before a form may be submitted.</dd>
  ),
  'aria-roledescription': (
    <dd>Defines a human-readable, author-localized description for the <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-role" className="internalDFN" data-link-type="dfn">role</a> of an <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element</a>.</dd>
  ),
  'aria-rowcount': (
    <dd>Defines the total number of rows in a <a href="https://www.w3.org/TR/wai-aria-1.1/#table" className="role-reference"><code>table</code></a>, <a href="https://www.w3.org/TR/wai-aria-1.1/#grid" className="role-reference"><code>grid</code></a>, or <a href="https://www.w3.org/TR/wai-aria-1.1/#treegrid" className="role-reference"><code>treegrid</code></a>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-rowindex" className="property-reference"><code>aria-rowindex</code></a>.</dd>
  ),
  'aria-rowindex': (
    <dd>Defines an <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-element" className="internalDFN" data-link-type="dfn">element's</a> row index or position with respect to the total number of rows within a <a href="https://www.w3.org/TR/wai-aria-1.1/#table" className="role-reference"><code>table</code></a>, <a href="https://www.w3.org/TR/wai-aria-1.1/#grid" className="role-reference"><code>grid</code></a>, or <a href="https://www.w3.org/TR/wai-aria-1.1/#treegrid" className="role-reference"><code>treegrid</code></a>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-rowcount" className="property-reference"><code>aria-rowcount</code></a> and <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-rowspan" className="property-reference"><code>aria-rowspan</code></a>.</dd>
  ),
  'aria-rowspan': (
    <dd>Defines the number of rows spanned by a cell or gridcell within a <a href="https://www.w3.org/TR/wai-aria-1.1/#table" className="role-reference"><code>table</code></a>, <a href="https://www.w3.org/TR/wai-aria-1.1/#grid" className="role-reference"><code>grid</code></a>, or <a href="https://www.w3.org/TR/wai-aria-1.1/#treegrid" className="role-reference"><code>treegrid</code></a>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-rowindex" className="property-reference"><code>aria-rowindex</code></a> and <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-colspan" className="property-reference"><code>aria-colspan</code></a>.</dd>
  ),
  'aria-selected': (
    <dd>Indicates the current "selected" <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-state" className="internalDFN" data-link-type="dfn">state</a> of various <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-widget" className="internalDFN" data-link-type="dfn">widgets</a>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-checked" className="state-reference"><code>aria-checked</code></a> and <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-pressed" className="state-reference"><code>aria-pressed</code></a>.</dd>
  ),
  'aria-setsize': (
    <dd>Defines the number of items in the current set of listitems or treeitems. Not required if all elements in the set are present in the <abbr title="Document Object Model">DOM</abbr>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-posinset" className="property-reference"><code>aria-posinset</code></a>.</dd>
  ),
  'aria-sort': (
    <dd>Indicates if items in a table or grid are sorted in ascending or descending order.</dd>
  ),
  'aria-valuemax': (
    <dd>Defines the maximum allowed value for a range <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-widget" className="internalDFN" data-link-type="dfn">widget</a>.</dd>
  ),
  'aria-valuemin': (
    <dd>Defines the minimum allowed value for a range <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-widget" className="internalDFN" data-link-type="dfn">widget</a>.</dd>
  ),
  'aria-valuenow': (
    <dd>Defines the current value for a range <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-widget" className="internalDFN" data-link-type="dfn">widget</a>. See related <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-valuetext" className="property-reference"><code>aria-valuetext</code></a>.</dd>
  ),
  'aria-valuetext': (
    <dd>Defines the human readable text alternative of <a href="https://www.w3.org/TR/wai-aria-1.1/#aria-valuenow" className="property-reference"><code>aria-valuenow</code></a> for a range <a href="https://www.w3.org/TR/wai-aria-1.1/#dfn-widget" className="internalDFN" data-link-type="dfn">widget</a>.</dd>
  ),
  'role': (
    <dd>Main indicator of type.  This <a className="termref internalDFN" data-lt="semantics" href="#dfn-semantics" data-link-type="dfn">semantic</a> association allows tools to present and support interaction with the object in a manner that is consistent with user expectations about other objects of that type.</dd>
  ),
};

interface Props {
  ariaProperty: string;
}

export const AriaDefinition = (props: Props) => {
  let ariaHash = props.ariaProperty.toLowerCase().replace('aria', 'aria-');
  let ariaUrl = WIA_ARIA_PATH + '#' + ariaHash;
  if (props.ariaProperty === 'role') {
    ariaUrl += '_definitions';
  }

  return (
    <Layout>
      <Layout margin={{ bottom: 0.5 }}>
        <CoreLink to={ariaUrl}>{ariaUrl}</CoreLink>
      </Layout>
      {AriaDefinitions[props.ariaProperty && ariaHash ? ariaHash : 'role']}
    </Layout>
  );
};
