import {
  Button,
  ButtonType,
  Display,
  FlexDirection,
  InjectStyledLayout,
  JustifyContent,
  Layout,
} from 'carbon-components-prototype';
import * as Prism from 'prismjs';
import * as React from 'react';
import * as CopyToClipboard from 'react-copy-to-clipboard';
import { renderToStaticMarkup } from 'react-dom/server';
import * as renderHTML from 'react-render-html';
import { trackCodeClickInteraction, trackCopyCodeButton } from '../../utils/analytics';
import './styles.scss';

// Add syntax highlighting support for additional languages.
import 'prismjs/components/prism-bash';
import 'prismjs/components/prism-jsx';
import 'prismjs/components/prism-sass';
import 'prismjs/components/prism-scss';
import 'prismjs/components/prism-yaml';

interface Props {
  children?: any; // tslint:disable-line no-any
  noCopy?: boolean;
  renderCode?: string;
  language?: string;
}

interface State {
  copied?: boolean;
  content?: string;
  clipboardInput?: HTMLTextAreaElement;
}

/**
 * Any code snippet should be implemented using this component.
 */
export class CodeEditor extends React.Component<Props, State> {
  constructor(props: Props) {
    super(props);

    this.state = {
      copied: false,
    };
  }

  public render() {
    let copyButton: JSX.Element | undefined;
    let renderCode = '';

    if (typeof window !== 'undefined') {
      let textContent = window.document.createRange().createContextualFragment(renderToStaticMarkup(this.props.children)).textContent;
      if (textContent) {
        renderCode = textContent.trim();
      }
    }

    if (!this.props.noCopy) {
      copyButton = (
        <Layout display={Display.Flex} justifyContent={JustifyContent.End} padding={{ top: 1 }}>
          <div>
            <CopyToClipboard text={renderCode} onCopy={this.handleCopy}>
              <Button type={ButtonType.Text}>{this.state.copied ? 'Copied' : 'Copy Code'}</Button>
            </CopyToClipboard>
          </div>
        </Layout>
      );
    }

    return (
      <Layout
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        className="code-editor"
      >
        <InjectStyledLayout
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
          flexGrow={1}
          elevation={2}
        >
          <pre
            onClick={trackCodeClickInteraction}
          >
            <code>
              {this.highlightCode(this.props.children, this.props.language)}
            </code>
          </pre>
        </InjectStyledLayout>
        <Layout display={Display.HideAccessible}>
          <textarea value={renderCode} readOnly />
        </Layout>
        {copyButton}
      </Layout>
    );
  }

  private highlightCode = (code: string, language?: string) => {
    let prismLanguage = language && Prism.languages[language];
    let renderCode = code;

    if (prismLanguage) {
      renderCode = Prism.highlight(code, prismLanguage, language);
    }

    return renderHTML(renderCode);
  }

  private handleCopy = () => {
    this.setState({ copied: true });

    trackCopyCodeButton();

    setTimeout(() => {
      this.setState({ copied: false });
    }, 2000);
  }
}
