import { Aspect, AspectRatio, Background, Layout, StyledLayout } from 'carbon-components-prototype';
import * as classnames from 'classnames';
import * as React from 'react';
import * as renderHTML from 'react-render-html';
import { DOMParser, XMLSerializer } from 'xmldom';
import * as YAML from 'yaml-js';
import { CodeEditor } from '../code-editor';
import { CodePreview } from '../code-preview';
import { BlockColumns } from './components/block-columns';
import { BlockGuidelines } from './components/block-guidelines';
import { BlockLinks } from './components/block-links';
import { BlockMedia } from './components/block-media';
import './styles.scss';

export enum BaseSize {
  Default = 1,
  Larger,
}

const BASE_SIZE_CLASSES = {
  [BaseSize.Default]: 'markdown--size-default',
  [BaseSize.Larger]: 'markdown--size-larger',
};

interface Props {
  source: string;
  size?: BaseSize;
  hideCopyCodeButton?: boolean;
}

/**
 * All markdown should be implemented using this component.
 */
export class Markdown extends React.Component<Props, {}> {

  public render() {
    if (!this.props.source) {
      return <div />;
    }

    const classes: ClassValue = {};
    let content: JSX.Element[] = [];

    if (this.props.size) {
      classes[BASE_SIZE_CLASSES[this.props.size]] = true;
    } else {
      classes[BASE_SIZE_CLASSES[BaseSize.Default]] = true;
    }

    // Because Gatsby runs these files on both the server and in the browser,
    // we need a way to parse the DOM without using the window object (that
    // is not availabe on the server). To do this, we use the `xmldom`
    // package to provide a parser and serializer.
    const DOM = this.parseFromString(this.props.source);
    const nodes = Array.from(DOM.childNodes);

    nodes.forEach((node: Element, index: number) => {
      if (!node.childNodes) {
        return;
      }

      const childNodes = Array.from(node.childNodes);
      const elem = node.nodeName.match(/pre/) ? node : childNodes.find((n) => !!(n.nodeName && n.nodeName.match(/pre/))) as Element;

      if (!elem) {
        content.push(this.renderMarkdownBlock(node, index));
        return;
      }

      if (this.getCodeLanguage(elem).match(/yml\sblock/)) {
        content.push(this.renderCustomBlock(elem, index));
      } else {
        content.push(this.renderCodeBlock(elem, index));
      }
    });

    return (
      <div className={`${classnames('markdown', classes)}`}>
        {content}
      </div>
    );
  }

  private renderCodeBlock = (elem: Element, key: number) => {
    const language = this.getCodeLanguage(elem);

    let showEditor = true;
    let showPreview = false;

    // If the language is `jsx` show the code preview, unless the
    // `hide-preview` is appened.
    if (language.match(/^jsx/) && !language.match(/hide\-preview/)) {
      showPreview = true;
    }

    // If there is a code preview, the source code can be hidden by appending
    // the `hide-code` class.
    if (language.match(/hide\-code/)) {
      showEditor = false;
    }

    let codeEditor = (
      <CodeEditor
        noCopy={!language || !!language.match(/no\-copy/) || this.props.hideCopyCodeButton}
        language={language.replace(/\s.*$/, '')}
      >
        {elem.textContent}
      </CodeEditor>
    );

    return (
      <Layout margin={{ bottom: 2 }} key={`markdown-${key}`} className="live-provider">
        {showPreview ? (
          <CodePreview renderCode={`<Layout fullWidth>${elem.textContent}</Layout>`}>
            {showEditor && codeEditor}
          </CodePreview>
        ) : codeEditor}
      </Layout>
    );
  }

  private renderCustomBlock = (elem: Element, key: number) => {
    const language = this.getCodeLanguage(elem);
    const yamlString = elem.textContent ? elem.textContent.replace(/^\s+/, '') : '';
    const data = YAML.load(yamlString);

    return (
      <Layout margin={{ bottom: 2 }} key={key}>
        {language.match(/block-links/) && <BlockLinks key={key} data={data} />}
        {language.match(/block-columns/) && <BlockColumns key={key} data={data} />}
        {language.match(/block-guideline/) && <BlockGuidelines key={key} data={data} />}
        {language.match(/block-media/) && <BlockMedia key={key} data={data} />}
      </Layout>
    );
  }

  private renderMarkdownBlock = (elem: Element, index: number) => {
    let html = this.serializeToString(elem);

    // If this element is a video embed iframe (from Google Drive, YouTube,
    // or Vimeo), wrap it in an Aspect component with a background.
    if (html.match(/\<iframe.*(drive\.google|youtube|vimeo).*\<\/iframe\>/)) {
      return (
        <StyledLayout background={Background.Alt} className="markdown__iframe" margin={{ bottom: 2 }}>
          <Aspect ratio={AspectRatio.Aspect16x9}>
            {renderHTML(html)}
          </Aspect>
        </StyledLayout>
      );
    }

    // Check for color variable references and append swatches next to them.
    const colorVarRefs = html.match(/\$(white|black|red|orange|yellow|green|blue|prime-blue|info|success|error|warn|opac-b|opac-w|twitch\-purple|hinted\-grey|color-(\w*))(\-[0-9]{1,2}){0,1}/g) || [];
    colorVarRefs.map((ref) => {
      html = html.replace(
        `<code>${ref}</code>`,
        `
          <div className="tw-inline-flex tw-align-items-center">
            <code data-replaced="true">${ref}</code>
            <div class="markdown__color-swatch markdown__color-swatch--${ref.replace('$', '')} tw-mg-l-05"></div>
          </div>
        `);
    });

    return (
      <Layout className="markdown__block" key={`markdown-${index}`}>
        {renderHTML(html)}
      </Layout>
    );
  }

  private getCodeLanguage = (elem: Element) => {
    let attribute: Attr | undefined = Array.from(elem.attributes).find((attr: Attr) => attr.name === 'data-language');

    if (!attribute && elem.firstChild) {
      attribute = Array.from((elem.firstChild as Element).attributes).find((attr: Attr) => attr.name === 'class');
    }

    return attribute ? attribute.value.replace(/language\-/, '') : '';
  }

  private serializeToString = (elem: Element) => {
    const serializer = new XMLSerializer();
    return serializer.serializeToString(elem);
  }

  private parseFromString = (source: string) => {
    const parser = new DOMParser();
    return parser.parseFromString(source, 'text/html');
  }

}
