import {
  AlignItems,
  Background,
  BorderRadius,
  Button,
  ButtonType,
  Column,
  ColumnOrder,
  CoreImage,
  CoreText,
  Display,
  FlexDirection,
  Grid,
  GridGutterSize,
  Layout,
  Overflow,
  Position,
  StyledLayout,
  SVGAsset,
  TextType,
  ZIndex,
} from 'carbon-components-prototype';
import * as React from 'react';

export interface BlockLink {
  title?: string;
  url?: string;
  thumbnailSrc?: string;
  body?: string;
  download?: boolean;
}

interface Props {
  data: BlockLink[];
}

export class BlockLinks extends React.Component<Props, {}> {
  public render() {
    const { data } = this.props;

    return (
      <Layout
        position={Position.Relative}
        padding={{ y: 3 }}
      >
        {this.renderBackground()}
        {data && data.length < 2 ? this.renderSingleLink(data[0]) : this.renderMultipleLinks(data)}
      </Layout>
    );
  }

  private renderBackground = () => (
    <Layout
      position={Position.Absolute}
      padding={{ x: 2 }}
      zIndex={ZIndex.Below}
      fullHeight
      fullWidth
      attachTop
    >
      <StyledLayout
        position={Position.Relative}
        background={Background.Alt}
        fullHeight
        fullWidth
      />
    </Layout>
  )

  private renderButton = (link: BlockLink) => {
    if (link.download) {
      return (
        <Button
          type={ButtonType.Hollow}
          icon={SVGAsset.Download}
          linkTo={link.url}
          download
        >
          Download
        </Button>
      );
    } else {
      return <Button linkTo={link.url}>Visit</Button>;
    }
  }

  private renderSingleLink = (link: BlockLink) => (
    <StyledLayout
      className="block-links"
      background={Background.Base}
      borderRadius={BorderRadius.Medium}
      overflow={Overflow.Hidden}
      elevation={3}
    >
      <Grid alignItems={AlignItems.Stretch}>
        <Column cols={{ default: 12, md: 6 }} order={{ default: ColumnOrder.Last, md: ColumnOrder.First }}>
          <Layout
            padding={2}
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            fullHeight
          >
            <Layout flexGrow={1}>
              <CoreText type={TextType.H4}>
                {link.title}
              </CoreText>
              <CoreText>
                {link.body}
              </CoreText>
            </Layout>
            <Layout margin={{ top: 2 }}>
              {this.renderButton(link)}
            </Layout>
          </Layout>
        </Column>
        {link.thumbnailSrc && (
          <Column cols={{ default: 12, md: 5 }} offset={{ default: 0, md: 1 }}>
            <CoreImage alt="" src={link.thumbnailSrc} />
          </Column>
        )}
      </Grid>
    </StyledLayout>
  )

  private renderMultipleLinks = (links: BlockLink[]) => (
    <Grid gutterSize={GridGutterSize.Medium}>
      {links.map((link) => (
        <Column cols={{ default: 12, sm: 6 }}>
          <StyledLayout
            className="block-links"
            background={Background.Base}
            borderRadius={BorderRadius.Medium}
            overflow={Overflow.Hidden}
            elevation={2}
          >
            {link.thumbnailSrc && <CoreImage alt="" src={link.thumbnailSrc} />}
            <Layout padding={2}>
              <CoreText type={TextType.H4}>
                {link.title}
              </CoreText>
              <CoreText>
                {link.body}
              </CoreText>

              <Layout margin={{ top: 2 }}>
                {this.renderButton(link)}
              </Layout>
            </Layout>
          </StyledLayout>
        </Column>
      ))}
    </Grid>
  )
}
