import {
  AlignItems,
  Column,
  Display,
  Grid,
  JustifyContent,
  Layout,
  StyledLayout,
  TabItemPublicProps,
  Tabs,
  TabSize,
  Tooltip,
  TooltipAlign,
} from 'carbon-components-prototype';
import GatsbyLink from 'gatsby-link';
import * as React from 'react';
import { RouteComponentProps, withRouter } from 'react-router-dom';
import { File, FileConnection, FileEdge } from '../../graphql-types';
import { sortByRelativePath } from '../../utils/sort-by-relative-path';
import './styles.scss';

interface PublicProps {
  page: File;
  tabs: FileConnection;
  isComponent?: boolean;
}

type Props = PublicProps & RouteComponentProps<{}>;

class PageTabsComponent extends React.Component<Props, {}> {
  constructor (props: Props) {
    super(props);
  }

  public render () {
    const tabs = this.getTabs();
    const activeTabIndex = this.props.location.pathname ? tabs.findIndex((item) => {
      return !!item.linkTo && this.props.location.pathname.replace(/\/$/, '').endsWith(item.linkTo.toString().replace(/\/$/, ''));
    }) : 0;

    return (
      <StyledLayout borderBottom className="page-tabs">
        <Grid>
          <Column cols={{ default: 6, md: 8 }}>
            <Tabs
              tabs={tabs}
              activeTabIndex={activeTabIndex}
              size={TabSize.Large}
              justifyContent={JustifyContent.Start}
              borderBottom={false}
            />
          </Column>

          {this.props.isComponent && (
            <Column offset={1} cols={{ default: 5, md: 3 }}>
              <Layout display={Display.Flex} alignItems={AlignItems.End} justifyContent={JustifyContent.End} fullHeight>
                {this.renderPlaygroundLink()}
              </Layout>
            </Column>
          )}
        </Grid>
      </StyledLayout>
    );
  }

  private getTabs = (): TabItemPublicProps[] => {
    const tabs: TabItemPublicProps[] = [];
    let { edges } = this.props.tabs;

    let indexTab = edges && edges.find((edge) => (
      edge &&
      edge.node &&
      edge.node.fields &&
      edge.node.fields.tab &&
      edge.node.fields.tab.slug === 'index' ||
      false
    ));

    if (
      indexTab &&
      indexTab.node &&
      indexTab.node.fields &&
      indexTab.node.fields.tabPath &&
      indexTab.node.fields.tab
    ) {
      tabs.push({
        label: this.getTitle(indexTab),
        linkTo: indexTab.node.fields.tabPath,
      });
    }

    sortByRelativePath(edges || []).forEach((edge) => {
      if (
        edge.node &&
        edge.node.fields &&
        edge.node.fields.tabPath &&
        edge.node.fields.tab &&
        edge.node.fields.tab.slug !== 'index'
      ) {
        tabs.push({
          label: this.getTitle(edge),
          linkTo: edge.node.fields.tabPath,
        });
      }
    });

    if (this.props.isComponent && this.props.page.fields && this.props.page.fields.path) {
      tabs.push({
        label: 'Properties',
        linkTo: `${this.props.page.fields.path}/properties`,
      });

      tabs.push({
        label: 'Changelog',
        linkTo: `${this.props.page.fields.path}/changelog`,
      });
    }

    return tabs;
  }

  private getTitle = (edge: FileEdge) => {
    if (
      edge.node &&
      edge.node.childMarkdownRemark &&
      edge.node.childMarkdownRemark.frontmatter &&
      edge.node.childMarkdownRemark.frontmatter.title
    ) {
      return edge.node.childMarkdownRemark.frontmatter.title;
    }

    if (
      edge.node &&
      edge.node.fields &&
      edge.node.fields.tab &&
      edge.node.fields.tab.title
    ) {
      return edge.node.fields.tab.title;
    }

    return '';
  }

  private renderPlaygroundLink = () => {
    let icon = (
      <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 20 20">
        <path d="M6 2h8a4 4 0 0 1 4 4v8a4 4 0 0 1-4 4H6a4 4 0 0 1-4-4V6a4 4 0 0 1 4-4zm0 2a2 2 0 0 0-2 2v8a2 2 0 0 0 2 2h8a2 2 0 0 0 2-2V6a2 2 0 0 0-2-2H6zm4 2a1 1 0 0 1 0 2H7a1 1 0 1 1 0-2h3zm3 3a1 1 0 0 1 0 2H9a1 1 0 0 1 0-2h4zm-3 3a1 1 0 0 1 0 2H7a1 1 0 0 1 0-2h3z"/>
      </svg>
    );

    if (!this.props.page.fields || !this.props.page.fields.path) {
      return undefined;
    }

    const playgroundSrc = `${this.props.page.fields.path}/playground/`;

    if (
      this.props.page.childMarkdownRemark &&
      this.props.page.childMarkdownRemark.frontmatter &&
      this.props.page.childMarkdownRemark.frontmatter.component &&
      this.props.page.childMarkdownRemark.frontmatter.component.disablePlayground
    ) {
      return (
        <Tooltip
          align={TooltipAlign.Right}
          label="Playground is not yet available for this component."
        >
          <GatsbyLink
            to={playgroundSrc}
            className="page-tabs__button page-tabs__button--disabled"
          >
            <Layout display={Display.Flex} alignItems={AlignItems.Center}>
              {icon} <Layout padding={{ left: 0.5 }}>Open in Playground</Layout>
            </Layout>
          </GatsbyLink>
        </Tooltip>
      );
    } else {
      return (
        <GatsbyLink
          to={playgroundSrc}
          className="page-tabs__button"
        >
          <Layout display={Display.Flex} alignItems={AlignItems.Center}>
            {icon} <Layout padding={{ left: 0.5 }}>Open in Playground</Layout>
          </Layout>
        </GatsbyLink>
      );
    }
  }
}

export const PageTabs: React.ComponentClass<PublicProps> = withRouter(PageTabsComponent);
