import {
  AlignItems,
  Background,
  BorderRadius,
  ButtonIcon,
  CoreText,
  Display,
  FlexDirection,
  FlexWrap,
  FontSize,
  JustifyContent,
  Layout,
  Overflow,
  Position,
  StyledLayout,
  SVGAsset,
  Tooltip,
  TooltipDirection,
} from 'carbon-components-prototype';
import { debounce } from 'lodash';
import * as queryString from 'query-string';
import * as React from 'react';
import * as ReactDOM from 'react-dom';
import { RouteComponentProps, withRouter } from 'react-router-dom';
import { CodeEditor } from '../../components/code-editor';
import { CodePreview } from '../../components/code-preview';
import { Overlay } from '../../components/overlay';
import { TypeDocFile } from '../../graphql-types';
import { ComponentPlaygroundProperty, PropField } from './components/prop-field';
import { propsToRenderString, setupDefaultPropValues } from './utils/property-helpers';

import './styles.scss';

interface PublicProps {
  file: TypeDocFile;
  onClickClose?: () => void;
}

interface State {
  props: ComponentPlaygroundProperty[];
  query?: { [key: string]: string };
  previewWidth?: number;
}

type Props = PublicProps & RouteComponentProps<{}>;

/**
 * This is the playground component.
 */
class PlaygroundComponent extends React.Component<Props, State> {
  private previewWrapperRef: Layout | null;
  private previewRef: HTMLDivElement | null;

  constructor(props: Props) {
    super(props);

    this.state = {
      props: setupDefaultPropValues(this.props.file.properties || []),
    };

    this.updateURLParams = debounce(this.updateURLParams, 500);
  }

  public componentDidMount () {
    this.updatePropsFromURL();
    window.addEventListener('resize', this.calculateAndSetPreviewSize);
  }

  public componentWillUnmount () {
    window.removeEventListener('resize', this.calculateAndSetPreviewSize);
  }

  public render () {
    let showOverlay: boolean = (undefined !== this.state.props.find((prop) => (
      prop.currentValue &&
      prop.type === 'boolean' &&
      prop.name === 'overlay' ||
      false
    )));

    const requiredPropertyList = this.state.props.filter((prop) => (prop.flags === null || !prop.flags.isOptional)).map((props, index) => (
      <PropField onValueChange={this.updatePreview} key={index} {...props} />
    ));

    const optionalPropertyList = this.state.props.filter((prop) => (prop.flags !== null && prop.flags.isOptional)).map((props, index) => (
      <PropField onValueChange={this.updatePreview} key={index} {...props} />
    ));

    return (
      <Overlay>
        <StyledLayout
          borderRadius={BorderRadius.Medium}
          overflow={Overflow.Hidden}
          fullHeight
        >
          <Layout display={Display.Flex} flexWrap={FlexWrap.NoWrap} className="playground">
            <StyledLayout background={Background.Base} position={Position.Absolute} fullHeight attachTop attachLeft display={Display.Flex} flexDirection={FlexDirection.Column} flexWrap={FlexWrap.NoWrap} flexShrink={0} className="playground__sidebar">
              <StyledLayout
                className="playground__header"
                display={Display.Flex}
                alignItems={AlignItems.Center}
                flexShrink={0}
                padding={{ x: 2, y: 1 }}
              >
                <Layout flexGrow={1}>
                  <CoreText fontSize={FontSize.Size4} bold>{this.props.file.name}</CoreText>
                </Layout>
                <Tooltip label="Reset Properties" direction={TooltipDirection.Bottom}>
                  <ButtonIcon
                    icon={SVGAsset.VideoRerun}
                    ariaLabel={`Reset`}
                    onClick={this.resetProps}
                  />
                </Tooltip>
              </StyledLayout>
              <Layout overflow={Overflow.Auto} padding={{ y: 2 }}>
                {requiredPropertyList}
                {optionalPropertyList}
              </Layout>
            </StyledLayout>
            <StyledLayout background={Background.Alt} display={Display.Flex} flexDirection={FlexDirection.Column} flexGrow={1} fullHeight className="playground__main">
              <StyledLayout
                className="playground__header"
                display={Display.Flex}
                alignItems={AlignItems.Center}
                flexShrink={0}
              >
                <Layout flexGrow={1} padding={1}>
                  {/* Add a quick share link. */}
                </Layout>
                <StyledLayout padding={1} borderLeft>
                  <ButtonIcon
                    onClick={this.props.onClickClose}
                    icon={SVGAsset.Close}
                    ariaLabel={`Back to ${this.props.file.name}`}
                  />
                </StyledLayout>
              </StyledLayout>
              <Layout
                padding={4}
                display={Display.Flex}
                position={Position.Relative}
                flexGrow={2}
                alignItems={AlignItems.Center}
                justifyContent={JustifyContent.Center}
                className="playground__preview"
              >
                <Layout
                  display={Display.Flex}
                  ref={(ref) => this.previewWrapperRef = ref}
                  fullWidth
                  fullHeight
                  className="playground__preview--content"
                >
                  <CodePreview
                    width={this.state.previewWidth}
                    onLoad={this.handlePreviewLoad}
                    refDelegate={this.getPreviewRef}
                    renderCode={propsToRenderString(this.props.file.name || 'NULL', this.state.props, false)}
                    showOverlay={showOverlay}
                    centerContent
                  />
                </Layout>
              </Layout>
              <Layout flexGrow={1} padding={{ bottom: 3, x: 4 }} className="playground__editor">
                <CodeEditor language="jsx">
                  {propsToRenderString(this.props.file.name || 'NULL', this.state.props, true)}
                </CodeEditor>
              </Layout>
            </StyledLayout>
          </Layout>
        </StyledLayout>
      </Overlay>
    );
  }

  private getPreviewRef = (ref: HTMLDivElement) => {
    this.previewRef = ref;
  }

  /**
   * Update the playground preview and code based on input.
   */
  private updatePreview = (name: string, value: string | boolean |null) => {
    // tslint:disable-next-line:no-any
    let query = this.state.query;
    let props = this.state.props;

    let updatedProp = props.find((p) => p.name === name);

    if (updatedProp) {
      updatedProp.currentValue = value;

      if (updatedProp.currentValue && updatedProp.name && query) {
        query[updatedProp.name] = updatedProp.currentValue.toString();
      } else if (updatedProp.name && query) {
        delete query[updatedProp.name];
      }
    }

    this.updateStateAndQuery(props, query);

    // Wait until the next frame before calculating the new preview size.
    window.requestAnimationFrame(this.calculateAndSetPreviewSize);
  }

  /**
   * This method gets the aspect ratio of the contents of the preview and
   * sets a width to ensure they will always fit vertically in the area.
   */
  private calculateAndSetPreviewSize = () => {
    if (!this.previewRef || !this.previewWrapperRef) {
      return;
    }

    const previewWrapperNode = ReactDOM.findDOMNode(this.previewWrapperRef);
    const previewNode = ReactDOM.findDOMNode(this.previewRef) as HTMLElement;
    const previewAspectRatio = previewNode.clientHeight / previewNode.clientWidth;

    this.setState({ previewWidth: previewWrapperNode.clientHeight / previewAspectRatio });
  }

  private handlePreviewLoad = () => {
    this.calculateAndSetPreviewSize();
  }

  /**
   * Reset the properties to the original values.
   */
  private resetProps = () => {
    let props = this.state.props;
    let query = this.state.query;

    props.map((prop) => {
      // Return the current value to the original value.
      prop.currentValue = prop.originalValue ? prop.originalValue : '';

      // Remove the query paramater from the URL.
      if (prop.name && query) {
        delete query[prop.name];
      }
    });

    this.updateStateAndQuery(props, query);
  }

  /**
   * A method to update both the state and URL query – they need to always
   * be mirrored.
   */
  // tslint:disable-next-line no-any
  private updateStateAndQuery = (props: any, query: any) => {
    this.updateURLParams(query);
    this.setState({ props, query });
  }

  private updatePropsFromURL = () => {
    let query = queryString.parse(this.props.history.location.search);
    let propertiesWithValues = setupDefaultPropValues(this.props.file.properties || []);

    // Set up the playground based on the query string:
    // Match keys and values from the query string.
    Object.keys(query).map((key) => {
      let prop = propertiesWithValues.find((p) => p.name === key);

      if (prop) {
        prop.currentValue = query[key];
      }
    });

    this.setState({
      props: propertiesWithValues,
      query,
    });
  }

  // tslint:disable-next-line no-any
  private updateURLParams = (query: any) => {
    this.props.history.replace({
      search: queryString.stringify(query),
    });
  }
}

export const Playground: React.ComponentClass<PublicProps> = withRouter(PlaygroundComponent);
