import { AlignItems, CoreText, Display, FontSize, Input, InputType, Layout, Select, Toggle } from 'carbon-components-prototype';
import * as React from 'react';
import { properties_3 as TypeDocFileProperty } from '../../../../graphql-types';
import { Code } from '../../../code';

import './styles.scss';

export interface ComponentPlaygroundProperty extends TypeDocFileProperty {
  currentValue: string | boolean | null;
  originalValue: string | boolean | null;
}

export interface Props extends ComponentPlaygroundProperty {
  onValueChange: (a: string, b: string|boolean|null) => void;
}

export class PropField extends React.Component<Props> {

  public render() {
    const isRequired = (this.props.flags === null || !this.props.flags.isOptional);
    return (
      <Layout
        className="prop-field"
        display={Display.Flex}
        alignItems={AlignItems.Center}
        key={this.props.name || ''}
        padding={{ x: 2, y: 1 }}
      >
        <Layout flexGrow={1}>
          <Layout padding={{ bottom: 0.5 }}>
            <CoreText fontSize={FontSize.Size6} bold>
              <Code required={isRequired}>
                {this.props.name}{!isRequired && '?'}
              </Code>
            </CoreText>
          </Layout>
          {this.props.type && (
            <CoreText fontSize={FontSize.Size7}>
              <Code>
                {this.props.type}
              </Code>
            </CoreText>
          )}
        </Layout>
        <Layout flexGrow={1} className="prop-field__input-wrapper">
          {this.renderFormField()}
        </Layout>
      </Layout>
    );
  }

  private renderFormField() {
    let inputCurrentValue = this.props.currentValue ? this.props.currentValue.toString() : '';

    // Set up the playground input – different types call for
    // different inputs.
    switch (this.props.type) {
      case 'React.ReactNode': {
        return (
          <Input
            type={InputType.Text}
            name={this.props.name || ''}
            value={inputCurrentValue}
            onChange={this.handleInput}
          />
        );
      }

      case 'string': {
        return (
          <Input
            type={InputType.Text}
            name={this.props.name || ''}
            value={inputCurrentValue}
            onChange={this.handleInput}
          />
        );
      }

      case 'number': {
        return (
          <Input
            type={InputType.Text}
            name={this.props.name || ''}
            value={inputCurrentValue}
            onChange={this.handleInput}
          />
        );
      }

      case 'boolean': {
        return (
          <Toggle
            checked={!!inputCurrentValue}
            name={this.props.name || ''}
            onChange={this.handleInput}
          />
        );
      }

      case 'BreakpointProps': {
        return undefined;
      }

      default: {
        if (this.props.availableValues && this.props.availableValues !== null) {
          return (
            <Select
              onChange={this.handleSelect}
              name={this.props.name || ''}
              value={inputCurrentValue}
            >
              {this.props.flags && this.props.flags.isOptional ? <option /> : ''}
              {this.props.availableValues.map((entry) => (
                entry.values && entry.values.map((value) => (
                  <option value={value} key={value}>{value}</option>
                ))
              ))}
            </Select>
          );
        }
      }
    }

    return undefined;
  }

  private handleSelect = (e: React.ChangeEvent<HTMLSelectElement>) => {
    let value: string | null = e.target.value.length > 0 ? e.target.value : null;
    this.props.onValueChange(e.target.name, value);
  }

  private handleInput = (e: React.ChangeEvent<HTMLInputElement>) => {
    // Check for the `checked` key provided by toggles; otherwise, send
    // the `value of the input.
    if (e.target.checked) {
      this.props.onValueChange(e.target.name, e.target.checked);
    } else {
      this.props.onValueChange(e.target.name, e.target.value);
    }
  }
}
