import {
  AlignItems,
  Background,
  BorderRadius,
  Color,
  CoreLink,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  InjectLayout,
  Layout,
  Pill,
  StyledLayout,
  SVG,
  SVGAsset,
  Tooltip,
} from 'carbon-components-prototype';
import * as React from 'react';
import { availableValues_2, properties_4, TypeDocFile } from '../../graphql-types';
import { AriaDefinition } from '../aria-definition';
import { Code } from '../code';
import './styles.scss';

interface Props {
  /** A TypeDoc file returned by a Gatsby GQL query.  */
  file: TypeDocFile;
}

/**
 * Renders a list of propertie from a component file.
 */
export function PropertiesList(props: Props) {
  let requiredProps: JSX.Element[] = [];
  let optionalProps: JSX.Element[] = [];
  let breakpointProps: JSX.Element[] = [];
  let ariaProps: JSX.Element[] = [];
  let propsRow: JSX.Element | undefined;

  if (props.file.properties) {
    props.file.properties.map((prop, index) => {
      let ariaDefintion: JSX.Element | undefined;

      if (prop.name && prop.name.search(/aria|role/) === 0) {
        ariaDefintion = (
          <StyledLayout fontSize={FontSize.Size6} margin={{ bottom: 0.5 }}>
            <AriaDefinition ariaProperty={prop.name}/>
          </StyledLayout>
        );
      }

      propsRow = (
        <tr key={`prop-row-${index}`} className={`${ prop.flags && prop.flags.isOptional ? 'props-table__row' : 'props-table__row--required' }`}>
          <td className="props-table__cell">
            <Layout display={Display.Flex} alignItems={AlignItems.Baseline} padding={{ bottom: 0.5 }}>
              <Code required={prop.flags && !prop.flags.isOptional ? true : false}>{prop.name}{prop.flags && prop.flags.isOptional ? '?' : ''}</Code>
              {prop.breakpointCompatible ? (
                <Layout padding={{ left: 0.5 }}>
                  <Tooltip label="Breakpoint Compatible">
                    <Pill label="!" />
                  </Tooltip>
                </Layout>
              ) : null}
            </Layout>
          </td>
          <td className="props-table__cell">
            <Code>{getTypeDisplay(prop.type, prop.sourceSrc)}</Code>
          </td>
          <td className="props-table__cell props-table__cell--description">
            {prop.comment && prop.comment.shortText && (
              <Layout margin={{ bottom: 2 }}>
                <CoreText>{prop.comment.shortText}</CoreText>
              </Layout>
            )}
            {ariaDefintion}
            {prop.availableValues && prop.availableValues.length > 0 && (
              <Layout>
                One of: <br />
                {prop.availableValues.map((entry, i) => {
                  return (
                    <div key={i}>
                      {prop.availableValues && prop.availableValues.length > 1 && (
                        <Layout display={Display.Block} margin={{ top: 2, bottom: 0.5 }}>
                          <Code>
                            <CoreText bold>
                              {entry.sourceSrc && <CoreLink to={entry.sourceSrc}>{entry.name}</CoreLink> || entry.name}
                            </CoreText>
                          </Code>
                        </Layout>
                      )}
                      {entry.values && entry.values.map((value, n) => (renderPropValueExample(value, n, entry)))}
                      {entry.properties && renderPropInterfaceList(entry.properties)}
                    </div>
                  );
                })}
              </Layout>
            )}
          </td>
        </tr>
      );

      if (prop.flags && !prop.flags.isOptional) {
        requiredProps.push(propsRow);
      } else if (prop.name && prop.name.search(/breakpoint/) === 0) {
        breakpointProps.push(propsRow);
      } else if (prop.name && prop.name.search(/aria|role/) === 0) {
        ariaProps.push(propsRow);
      } else {
        optionalProps.push(propsRow);
      }

    });
  }

  return (
    <InjectLayout fullWidth>
      <table className="props-table">
        <tbody>
          <tr>
            <th className="props-table__heading">Property</th>
            <th className="props-table__heading">Type</th>
            <th className="props-table__heading">Description</th>
          </tr>
          {requiredProps}
          {optionalProps}
          {breakpointProps}
          {ariaProps}
        </tbody>
      </table>
    </InjectLayout>
  );
}

function getTypeDisplay(type: string|null, sourceSrc: string|null) {
  if (!type) {
    return;
  }

  if (!sourceSrc || ['number', 'string', 'boolean'].includes(type)) {
    return type;
  }

  return <CoreLink to={sourceSrc}>{type}</CoreLink>;
}

function renderPropInterfaceList(data: properties_4[]) {
  return (
    <Code>
      <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
        <div>&#123;</div>
        {data.map((item, i) => (
          <Layout padding={{ left: 2 }} key={i}>
            {item.name}{item.isOptional && '?'}:&nbsp;{item.sourceSrc ? <CoreLink to={item.sourceSrc}>{item.value}</CoreLink> : item.value};
          </Layout>
        ))}
        <div>&#125;</div>
      </Layout>
    </Code>
  );
}

function renderPropValueExample(value: string, index: number, propertyData: availableValues_2) {
  switch (propertyData.name) {

    case 'Background':
      return (
        <Layout display={Display.InlineFlex} alignItems={AlignItems.Center} key={index} margin={{ right: 1, bottom: 1 }}>
          <StyledLayout background={Background[(value as keyof typeof Background)]} border display={Display.InlineFlex} padding={1} margin={{ right: 0.5 }} />
          <Layout>
            <Code>{value}</Code>
            {propertyData.values && index < propertyData.values.length - 1 && ', '}
          </Layout>
        </Layout>
      );

    case 'Color':
      return (
        <Layout display={Display.InlineFlex} alignItems={AlignItems.Center} key={index} margin={{ right: 1, bottom: 1 }}>
          <StyledLayout color={Color[(value as keyof typeof Color)]} border display={Display.InlineFlex} margin={{ right: 0.5 }}>
            <div style={{ background: 'currentColor' }}>
              <Layout padding={1} />
            </div>
          </StyledLayout>
          <Layout>
            <Code>{value}</Code>
            {propertyData.values && index < propertyData.values.length - 1 && ', '}
          </Layout>
        </Layout>
      );

    case 'BorderRadius':
      return (
        <Layout display={Display.InlineFlex} alignItems={AlignItems.Center} key={index} margin={{ right: 1, bottom: 1 }}>
          <StyledLayout borderRadius={BorderRadius[(value as keyof typeof BorderRadius)]} border padding={2} background={Background.Alt2} display={Display.InlineFlex} margin={{ right: 0.5 }} />
          <Layout>
            <Code>{value}</Code>
            {propertyData.values && index < propertyData.values.length - 1 && ', '}
          </Layout>
        </Layout>
      );

    case 'FontSize':
      return (
        <Layout display={Display.Flex} alignItems={AlignItems.Center} key={index} margin={{ right: 1, bottom: 1 }}>
          <Layout>
            <Code>{value}</Code>:
          </Layout>
          <StyledLayout fontSize={FontSize[(value as keyof typeof FontSize)]} border background={Background.Alt2} display={Display.InlineFlex} padding={{ x: 1 }} margin={{ left: 0.5 }}>
            Hello World
          </StyledLayout>
        </Layout>
      );

    case 'SVGAsset':
      return (
        <Layout display={Display.InlineFlex} alignItems={AlignItems.Center} key={index} margin={{ right: 1, bottom: 0.5 }}>
          <StyledLayout display={Display.InlineFlex} margin={{ right: 0.5 }}>
            <SVG asset={SVGAsset[(value as keyof typeof SVGAsset)]} />
          </StyledLayout>
          <Layout>
            <Code>{value}</Code>
            {propertyData.values && index < propertyData.values.length - 1 && ', '}
          </Layout>

        </Layout>
      );

    default:
      return (
        <span key={index}>
          <Code>{value}</Code>
          {propertyData.values && index < propertyData.values.length - 1 && ', '}
        </span>
      );
  }
}
