import {
  Animation,
  AnimationType,
  Color,
  CoreText,
  Display,
  Layout,
  Pill,
  PillType,
  Position,
  StyledLayout,
} from 'carbon-components-prototype';
import * as React from 'react';
import './styles.scss';

interface Props {
  term: string;
  autoFocus?: boolean;
  onChange?: (searchTerm: string ) => void;
}

interface State {
  showKeyBindings: boolean;
  term: string;
}

export class SearchInput extends React.Component<Props, State> {
  private inputRef: HTMLInputElement | null;

  constructor (props: Props) {
    super(props);

    this.state = {
      term: props.term,
      showKeyBindings: false,
    };
  }

  public componentDidMount () {
    if (this.inputRef) {
      this.inputRef.focus();
    }
  }

  public componentWillReceiveProps (nextProps: Props) {
    this.setState({
      term: nextProps.term,
      showKeyBindings: nextProps.term.length > 0 ? true : false,
    });
  }

  public render() {
    return (
      <Layout
        position={Position.Relative}
        padding={{ bottom: 3 }}
        className="search-input"
      >
        <input
          type="text"
          className="search-input__input"
          placeholder="Search"
          onChange={this.onChange}
          onKeyDown={this.onKeyDown}
          ref={(ref) => this.inputRef = ref}
          value={this.props.term}
        />
        {this.state.showKeyBindings && (
          <StyledLayout
            className="search-input__meta"
            padding={{ x: 2 }}
            color={Color.Overlay}
            position={Position.Absolute}
            attachBottom
            attachLeft
            fullWidth
          >
            <Animation type={AnimationType.SlideInTop} enabled>
              <Layout display={Display.Flex}>
                <CoreText>Use <Pill type={PillType.Overlay} label="↓" /> <Pill type={PillType.Overlay} label="↑" /> to navigate results.</CoreText>
                <Layout flexGrow={1} margin={{ left: 1 }}>
                  <CoreText><Pill type={PillType.Overlay} label="enter" /> to select.</CoreText>
                </Layout>

                <CoreText><Pill type={PillType.Overlay} label="esc" /> to clear search.</CoreText>
              </Layout>
            </Animation>
          </StyledLayout>
        )}
      </Layout>
    );
  }

  private onKeyDown = (e: React.KeyboardEvent<HTMLInputElement>) => {
    // Prevent the cursor from moving to the front of the search input when
    // navigating through search inputs using the keyboard arrows.
    if (e.key === 'ArrowUp') {
      e.preventDefault();
      return;
    }

  }

  private onChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (this.props.onChange) {
      this.props.onChange(e.currentTarget.value);
    }
  }

}
