import { Layout } from 'carbon-components-prototype';
import { navigateTo } from 'gatsby-link';
import { debounce } from 'lodash';
import * as queryString from 'query-string';
import * as React from 'react';
import { RouteComponentProps, withRouter } from 'react-router-dom';
import { trackSearchResultPosition } from '../../../utils/analytics';
import { NoResults } from './../no-results';
import { SearchInput } from './../search-input';
import { SearchResult } from './../search-result';
import { FuseMatch, search, SearchItem } from './../utils/search-utils';
import './styles.scss';

const TRACKING_DEBOUNCE_TIMEOUT = 1000;

export interface PublicProps {
  items: SearchItem[];
  numResults: number;
  onChange?: (searchTerm: string) => void;
}

type Props = PublicProps & RouteComponentProps<{}>;

export interface State {
  focusIndex: number;
  searchTerm: string;
  matches: FuseMatch[];
}

class SearchComponent extends React.Component<Props, State> {
  constructor (props: Props) {
    super(props);

    const query = queryString.parse(props.location.search);
    const searchTerm = query && query.search || '';

    if (this.props.onChange) {
      this.props.onChange(searchTerm);
    }

    this.state = {
      focusIndex: -1,
      matches: this.getMatches(searchTerm),
      searchTerm: query && query.search || '',
    };
  }

  public render() {
    return (
      <div
        className="search"
        onKeyDown={this.onKeyDown}
        tabIndex={0}
      >
        <SearchInput
          autoFocus
          term={this.state.searchTerm}
          onChange={this.onInputChange}
        />

        {this.state.searchTerm.length > 0 && (
          <Layout padding={{ top: 5 }} />
        )}

        {this.state.matches.length > 0 && this.state.matches.map((item, index) => (
          <SearchResult
            onClick={this.trackSearchResultSelected}
            isFocused={this.state.focusIndex === index}
            item={item}
            index={index}
            key={index}
          />
        ))}

        {this.state.searchTerm.length > 0 && this.state.matches.length === 0 && (
          <NoResults />
        )}

      </div>
    );
  }

  private onInputChange = (searchTerm: string) => {

    // Set the search term.
    this.setSearchTerm(searchTerm);

    this.setState({
      focusIndex: 0,
      matches: this.getMatches(searchTerm),
    });
  }

  private getMatches = (searchTerm: string) => {
    return searchTerm.length > 0 ? search(searchTerm, this.props.items).slice(0, this.props.numResults) : [] as FuseMatch[];
  }

  private onKeyDown = (e: React.KeyboardEvent<HTMLDivElement>) => {
    if (e.key === 'ArrowDown' && this.state.focusIndex < this.state.matches.length - 1) {
      this.setState({ focusIndex: this.state.focusIndex + 1 });
    }
    if (e.key === 'ArrowUp' && this.state.focusIndex > 0) {
      this.setState({ focusIndex: this.state.focusIndex - 1 });
    }
    if (e.key === 'Enter' && this.state.focusIndex > -1) {
      // Find and track the search result.
      const currentMatch = this.state.matches[this.state.focusIndex];
      this.trackSearchResultSelected(this.state.focusIndex);

      // Navigate to the page.
      navigateTo(currentMatch.item.path);
    }
    if (e.key === 'Escape') {
      this.clearSearch();
    }
  }

  private trackSearchResultSelected = (index: number) => {
    // Send before we navigate to a result page.
    this.debounceTrackSearchTerm.flush();

    // Find and track the result.
    const currentMatch = this.state.matches[index];
    if (currentMatch) {
      trackSearchResultPosition(this.state.searchTerm, currentMatch.item.path, index);
    }
  }

  private clearSearch = () => {
    // Reset the search term.
    this.setSearchTerm('');

    // Clear the matches array.
    this.setState({ matches: [] });
  }

  private setSearchTerm = (searchTerm: string) => {
    this.setState({
      searchTerm,
    });

    this.debounceTrackSearchTerm();

    if (this.props.onChange) {
      this.props.onChange(searchTerm);
    }
  }

  // tslint:disable-next-line:member-ordering
  private debounceTrackSearchTerm = debounce(() => {

    // Don't replace history state if there is no change
    const currentQuery = queryString.parse(window.location.search);
    if (currentQuery.search === undefined && this.state.searchTerm === '') {
      return;
    }

    // Do use an empty string if there is no search term
    let term = '';

    if (this.state.searchTerm) {
      term = queryString.stringify({
        search: this.state.searchTerm,
      });
    }

    // This will automatically trigger a new Google Analytics pageview with URL param for search tracking
    this.props.history.replace({
      search: term,
    });

  }, TRACKING_DEBOUNCE_TIMEOUT);
}

export const Search: React.ComponentClass<PublicProps> = withRouter(SearchComponent);
