import {
  AlignItems,
  BorderRadius,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  Interactable,
  InteractableType,
  Layout,
  Overflow,
  StyledLayout,
  TextType,
} from 'carbon-components-prototype';
import { withPrefix } from 'gatsby-link';
import * as React from 'react';
import { RouteComponentProps, withRouter } from 'react-router-dom';
import { File, FileEdge } from '../../graphql-types';
import { getDisplayTitle } from '../../utils/display-titles';
import { sortByRelativePath } from '../../utils/sort-by-relative-path';
import './styles.scss';

interface PublicProps {
  pages?: FileEdge[] | null;
}

type Props = PublicProps & RouteComponentProps<{}>;

class SidebarMenuComponent extends React.Component<Props, {}> {
  constructor(props: Props) {
    super(props);
  }

  public render() {
    return (
      <StyledLayout
        className="sidebar-menu"
        display={Display.Flex}
        fontSize={FontSize.Size5}
        borderRight
        fullHeight
      >
        {this.renderMenuItems()}
      </StyledLayout>
    );
  }

  private renderMenuItems = () => {
    let sectionHeading: JSX.Element | undefined;
    let menuItems: JSX.Element[] = [];

    if (!this.props.pages) {
      return;
    }

    let currentSection = this.props.pages.find((page) => (
      page.node &&
      page.node.fields &&
      page.node.fields.segments &&
      page.node.fields.segments[0] &&
      page.node.fields.segments[0].slug === this.getCurrentPathSegments()[0] &&
      page.node.fields.segments.length === 1 ||
      false
    ));

    let currentSectionPages = this.props.pages.filter((page) => (
      currentSection &&
      currentSection.node &&
      currentSection.node.fields &&
      currentSection.node.fields.segments &&
      page.node &&
      page.node.fields &&
      page.node.fields.segments &&
      page.node.fields.segments[0] &&
      page.node.fields.segments[0].slug === currentSection.node.fields.segments[0].slug &&
      page.node.fields.segments.length > 1
    ));

    if (
      currentSection &&
      currentSection.node &&
      currentSection.node.fields &&
      currentSection.node.fields.segments &&
      currentSection.node.fields.title
    ) {

      // Add the menu heading.
      sectionHeading = (
        <Layout
          className="sidebar-menu__heading-bar"
          display={Display.Flex}
          alignItems={AlignItems.Center}
          padding={1}
        >
          <Layout padding={{ x: 1 }}>
            <CoreText
              type={TextType.H4}
              bold
            >
              {getDisplayTitle(currentSection.node)}
            </CoreText>
          </Layout>
        </Layout>
      );

      let subsection = currentSectionPages.filter((section) => (
        section.node &&
        section.node.fields &&
        section.node.fields.segments &&
        section.node.fields.segments.length === 2
      ));

      subsection.forEach((section, index) => {
        if (!section.node || !section.node.fields || !section.node.fields.title) {
          return;
        }

        let subsectionPages = currentSectionPages.filter((page) => (
          page.node &&
          page.node.fields &&
          page.node.fields.segments &&
          page.node.fields.segments[1] &&
          section.node &&
          section.node.fields &&
          section.node.fields.segments &&
          section.node.fields.segments[1] &&
          page.node.fields.segments[1].slug === section.node.fields.segments[1].slug &&
          page.node.fields.segments.length === 3
        ));

        // If this is the only page, return a link.
        if (subsectionPages.length === 0) {
          menuItems.push(<div key={`menu-item-${index}`}>{this.renderMenuItem(section)}</div>);
          return;
        }

        // Otherwise, continue through the pages within this "subsection".
        // Add a subsection title.
        menuItems.push(
          <Layout key={`menu-item-${index}`} padding={{ top: 2, x: 1, bottom: 0.5 }}>
            <CoreText
              color={Color.Alt2}
              fontSize={FontSize.Size5}
              className="sidebar-menu__menu-item-heading"
            >
              {getDisplayTitle(section.node)}
            </CoreText>
          </Layout>,
        );

        subsectionPages.forEach((page, n) => {

          // Get the children of the subsection.
          let children = sortByRelativePath(currentSectionPages).filter((child) => (
            page.node &&
            page.node.fields &&
            page.node.fields.segments &&
            page.node.fields.segments[1] &&
            page.node.fields.segments[2] &&
            child.node &&
            child.node.fields &&
            child.node.fields.segments &&
            child.node.fields.segments[1] &&
            child.node.fields.segments[2] &&
            child.node.fields.segments[1].slug === page.node.fields.segments[1].slug &&
            child.node.fields.segments[2].slug === page.node.fields.segments[2].slug &&
            child.node.fields.segments.length > 3
          ));

          // Add the link.
          menuItems.push(<div key={`menu-item-${index}-${n}`}>{this.renderMenuItem(page, children)}</div>);
        });

      });
    }

    return (
      <Layout
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        fullWidth
        fullHeight
      >
        <StyledLayout borderBottom>
          {sectionHeading}
        </StyledLayout>
        <Layout overflow={Overflow.Auto}>
          <Layout padding={{ x: 1, y: 2 }}>
            {menuItems}
          </Layout>
        </Layout>
      </Layout>
    );
  }

  private renderMenuItem = (page: FileEdge, children?: FileEdge[]) => {
    if (!page || !page.node || !page.node.fields || !page.node.fields.path || !page.node.fields.segments) {
      return <div />;
    }

    return (
      <Layout>

        {this.renderMenuLink(page.node)}

        {children && children.map((child, index) => {
          if (
            child &&
            child.node &&
            child.node.fields &&
            child.node.fields.path
          ) {
            return (
              <Layout key={`child-link-${index}`} padding={{ left: 1 }}>
                {this.renderMenuLink(child.node, true)}
              </Layout>
            );
          }
        })}
      </Layout>
    );
  }

  private renderMenuLink = (node: File, sub?: boolean) => {
    if (!node.fields || !node.fields.path) {
      return;
    }
    return (
      <div className="sidebar-menu__menu-item">
        <Interactable
          type={InteractableType.Alpha}
          linkTo={withPrefix(node.fields.path)}
          borderRadius={BorderRadius.Medium}
          selected={this.props.location.pathname.indexOf(node.fields.path) > -1}
          blurAfterClick
        >
          <Layout padding={{ x: 1, y: 0.5 }}>
            <CoreText
              type={TextType.Span}
              fontSize={sub ? FontSize.Size6 : FontSize.Size5}
            >
              {getDisplayTitle(node)}
            </CoreText>
          </Layout>
        </Interactable>
      </div>
    );
  }

  private getCurrentPath = () => {
    return this.props.history.location.pathname.replace(withPrefix(''), '').replace(/\/$/, '');
  }

  private getCurrentPathSegments = () => {
    return this.getCurrentPath().replace(/^\//, '').split('/');
  }
}

export const SidebarMenu: React.ComponentClass<PublicProps> = withRouter(SidebarMenuComponent);
