import {
  AlignItems,
  ButtonIcon,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  InjectLayout,
  InjectStyledLayout,
  JustifyContent,
  Layout,
  Overflow,
  Position,
  StyledLayout,
  SVG,
  SVGAsset,
  SVGType,
} from 'carbon-components-prototype';
import * as classnames from 'classnames';
import GatsbyLink, { withPrefix } from 'gatsby-link';
import * as React from 'react';
import { RouteComponentProps, withRouter } from 'react-router-dom';
import { FileEdge } from '../../graphql-types';
import { sortByRelativePath } from '../../utils/sort-by-relative-path';
import { VersionNumber } from './components/version-number';
import './styles.scss';

interface PublicProps {
  minimal?: boolean;
  pages?: FileEdge[] | null;
  currentProjectVersion: string;
}

type Props = PublicProps & RouteComponentProps<{}>;

/**
 * Main site sidebar.
 */
class SidebarComponent extends React.Component<Props, {}> {
  constructor(props: Props) {
    super(props);
  }

  public render() {
    const classes: ClassValue = {
      'sidebar': true,
      'sidebar--minimal': this.getCurrentPath() === '',
    };

    return (
      <Layout
        className={classnames(classes)}
        flexShrink={0}
        position={Position.Relative}
      >
        <StyledLayout
          className="sidebar__inner"
          display={Display.Flex}
          position={Position.Fixed}
          fullHeight
          borderRight
        >
          <Layout
            className="sidebar__anchor"
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            alignItems={AlignItems.Center}
            flexShrink={0}
            fullHeight
          >
            <Layout
              display={Display.Flex}
              alignItems={AlignItems.Start}
              justifyContent={JustifyContent.Center}
              flexGrow={0}
              fullWidth
              className="sidebar__heading-bar"
            >
              <InjectStyledLayout color={Color.Overlay} padding={1} className="sidebar__brand">
                <GatsbyLink to="/">
                  <SVG asset={SVGAsset.LogoGlitch} type={SVGType.Inherit} width={30} height={30} />
                </GatsbyLink>
              </InjectStyledLayout>
            </Layout>
            <Layout flexGrow={1} flexShrink={1} />
            <VersionNumber
              value={this.props.currentProjectVersion}
              linkTo={`https://git-aws.internal.justin.tv/core-ui/core-ui/releases/tag/v${this.props.currentProjectVersion}`}
            />
            <InjectStyledLayout padding={1} color={Color.Overlay}>
              <a href="https://git-aws.internal.justin.tv/core-ui/core-ui" title="Core UI on Github">
                <SVG asset={SVGAsset.Github} width={20} height={20} />
              </a>
            </InjectStyledLayout>
          </Layout>

          <Layout
            className="sidebar__menu"
            flexGrow={1}
            fullHeight
          >
            {this.renderMenuItems()}
          </Layout>
        </StyledLayout>
      </Layout >
    );
  }

  private renderMenuItems = () => {
    let menuHeading: JSX.Element | undefined;
    let menuItems: JSX.Element[] = [];

    if (!this.props.pages) {
      return;
    }

    let currentSection: FileEdge | undefined;
    let currentSectionDepth: number = 2;
    let currentSectionPages: FileEdge[] = this.getSectionPagesAtDepth(2);

    // If there are no pages at the current depth, try the next level up.
    if (currentSectionPages.length === 0) {
      currentSectionDepth = 1;
      currentSectionPages = this.getSectionPagesAtDepth(1);
    }

    // If there are no pages at this level, don't render a menu.
    if (currentSectionPages.length === 0) {
      return;
    }

    currentSection = this.getSectionAtDepth(currentSectionDepth);

    if (
      currentSection &&
      currentSection.node &&
      currentSection.node.fields &&
      currentSection.node.fields.segments &&
      currentSection.node.fields.title &&
      currentSectionPages
    ) {

      // Add the menu heading.
      menuHeading = this.renderMenuHeading(currentSection, currentSectionDepth);

      // Add the menu items.
      sortByRelativePath(currentSectionPages.filter((page) => (
        page &&
        page.node &&
        page.node.fields &&
        page.node.fields.segments &&
        page.node.fields.segments.length === currentSectionDepth + 1
      ))).forEach((item) => {

        let children = sortByRelativePath(currentSectionPages.filter((child) => (
          item.node &&
          item.node.fields &&
          item.node.fields.segments &&
          item.node.fields.segments[currentSectionDepth] &&
          child.node &&
          child.node.fields &&
          child.node.fields.segments &&
          child.node.fields.segments[currentSectionDepth] &&
          child.node.fields.segments[currentSectionDepth].slug === item.node.fields.segments[currentSectionDepth].slug &&
          child.node.fields.segments.length > currentSectionDepth + 1
        )));

        menuItems.push(this.renderMenuItem(item, currentSectionDepth, children));
      });
    }

    return (
      <Layout
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        fullHeight
      >
        <StyledLayout borderBottom>
          {menuHeading}
        </StyledLayout>
        <Layout overflow={Overflow.Auto}>
          <Layout padding={1}>
            {menuItems}
          </Layout>
        </Layout>
      </Layout>
    );
  }

  private getSectionPagesAtDepth = (depth: number) => {
    if (!this.props.pages) {
      return [];
    }

    return this.props.pages.filter((page) => (
      page.node &&
      page.node.fields &&
      page.node.fields.segments &&
      page.node.fields.segments[depth - 1] &&
      page.node.fields.segments[depth - 1].slug === this.getCurrentPathSegments()[depth - 1] &&
      page.node.fields.segments.length > depth
    ));
  }

  private getSectionAtDepth = (depth: number) => {
    if (!this.props.pages) {
      return undefined;
    }

    return this.props.pages.find((page) => (
      page.node &&
      page.node.fields &&
      page.node.fields.segments &&
      page.node.fields.segments[depth - 1] &&
      page.node.fields.segments[depth - 1].slug === this.getCurrentPathSegments()[depth - 1] &&
      page.node.fields.segments.length === depth ||
      false
    ));
  }

  private renderMenuItem = (page: FileEdge, depth: number, children?: FileEdge[]) => {
    if (!page || !page.node || !page.node.fields || !page.node.fields.path || !page.node.fields.segments) {
      return <div />;
    }

    return (
      <Layout>
        <InjectLayout display={Display.Flex} alignItems={AlignItems.Center}>
          <GatsbyLink
            className="sidebar__menu-item"
            activeClassName="sidebar__menu-item--active"
            to={page.node.fields.path}
          >
            <Layout flexGrow={1}>
              {page.node.fields.title}
            </Layout>
            {children && children.length > 0 && (
              <SVG asset={page.node.fields.segments.length > 2 ? SVGAsset.GlyphArrDown : SVGAsset.GlyphArrRight} width={10} />
            )}
          </GatsbyLink>
        </InjectLayout>
        {children && depth > 1 && children.map((child) => {
          if (
            child &&
            child.node &&
            child.node.fields &&
            child.node.fields.path
          ) {
            return (
              <Layout padding={{ left: 1 }}>
                <InjectLayout display={Display.Flex}>
                  <GatsbyLink
                    className="sidebar__menu-item sidebar__menu-item--sub"
                    activeClassName="sidebar__menu-item--active"
                    to={child.node.fields.path}
                  >
                    <Layout flexGrow={1}>
                      {child.node.fields.title}
                    </Layout>
                  </GatsbyLink>
                </InjectLayout>
              </Layout>
            );
          }
        })}
      </Layout>
    );

  }

  private renderMenuHeading = (section: FileEdge, depth?: number) => {
    return (
      <Layout

        className="sidebar__heading-bar"
        display={Display.Flex}
        alignItems={AlignItems.Center}
        padding={1}
      >
       {depth && depth > 1 && (
          <ButtonIcon
            icon={SVGAsset.GlyphArrLeft}
            linkTo={
              section.node && section.node.fields && section.node.fields.path ? section.node.fields.path.substr(0, section.node.fields.path.lastIndexOf('/')) : ''
            }
            ariaLabel={`Go Back`}
          />
        )}
        <Layout padding={{ x: 1 }}>
          <CoreText fontSize={FontSize.Size4} bold>{section.node && section.node.fields && section.node.fields.title}</CoreText>
        </Layout>
      </Layout>
    );
  }

  private getCurrentPath = () => {
    return this.props.history.location.pathname.replace(withPrefix(''), '').replace(/\/$/, '');
  }

  private getCurrentPathSegments = () => {
    return this.getCurrentPath().replace(/^\//, '').split('/');
  }
}

export const Sidebar: React.ComponentClass<PublicProps> = withRouter(SidebarComponent);
