import {
  AlignItems,
  Avatar,
  Color,
  Column,
  CoreText,
  Display,
  FlexDirection,
  Grid,
  InjectStyledLayout,
  Layout,
  Overflow,
  Position,
  TextType,
  ZIndex,
} from 'carbon-components-prototype';
import { withPrefix } from 'gatsby-link';
import * as React from 'react';
import { Helmet } from 'react-helmet';
import { FileConnection, SitePageConnection, TypeDocFileConnection } from 'src/graphql-types';
import { Search } from '../components/search/search/component';
import { dataToSearchObjects } from '../components/search/utils/search-utils';
import { TileLink } from '../components/tile-link';
import './styles.scss';

export interface Props {
  data: {
    allPages: SitePageConnection;
    allFiles: FileConnection;
    allTypeDocs: TypeDocFileConnection;
  };
}

interface State {
  searchActive: boolean;
}

const enum TileThumb {
  Web = 1,
  Brand,
}

const tileThumbPaths = {
  [TileThumb.Brand]: <svg xmlns="http://www.w3.org/2000/svg" width="36" height="36" viewBox="0 0 36 36"><path fill="currentColor" fillRule="evenodd" d="M12 27a9 9 0 1 1 4.334-16.89A10.968 10.968 0 0 0 13 18c0 3.095 1.278 5.89 3.334 7.89A8.96 8.96 0 0 1 12 27zm6 .221A10.949 10.949 0 0 1 12 29C5.925 29 1 24.075 1 18S5.925 7 12 7c2.214 0 4.274.654 6 1.779A10.949 10.949 0 0 1 24 7c6.075 0 11 4.925 11 11s-4.925 11-11 11a10.95 10.95 0 0 1-6-1.779zm1.666-17.11a9 9 0 1 1 0 15.779A10.967 10.967 0 0 0 23 18c0-3.095-1.278-5.89-3.334-7.89zM18 11.29A8.977 8.977 0 0 1 21 18a8.977 8.977 0 0 1-3 6.708A8.977 8.977 0 0 1 15 18a8.977 8.977 0 0 1 3-6.708z" clipRule="evenodd"/></svg>,
  [TileThumb.Web]: <svg xmlns="http://www.w3.org/2000/svg" width="36" height="36" viewBox="0 0 36 36"><path fill="currentColor" fillRule="evenodd" d="M31 8H5v4h26V8zM5 28V14h26v14H5zM5 6h26a2 2 0 0 1 2 2v20a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2zm2 5a1 1 0 1 0 0-2 1 1 0 0 0 0 2zm4-1a1 1 0 1 1-2 0 1 1 0 0 1 2 0zm2 1a1 1 0 1 0 0-2 1 1 0 0 0 0 2z" clipRule="evenodd"/></svg>,
};

// Gatsby requires a default export for template and layout files.
// tslint:disable-next-line:no-default-export
export default class HomepageTemplate extends React.Component<Props, State> {

  constructor(props: Props) {
    super(props);

    this.state = {
      searchActive: false,
    };
  }

  // export default function HomepageTemplate(props: Props) {
  public render() {
    return (
      <Layout
        padding={{ x: 5, bottom: 5 }}
        position={Position.Relative}
        className="landing-page"
      >

        <Helmet>
          <title>{process.env.SITE_METADATA.title}</title>
        </Helmet>

        <div className="landing-page__background" />

        <Layout
          position={Position.Relative}
          padding={{ y: 4 }}
          className="landing-page__wrapper"
        >
          <Grid>
            <Column cols={{ default: 12, sm: 10, md: 8 }}>
              <Search
                numResults={7}
                items={dataToSearchObjects(this.props.data.allPages, this.props.data.allFiles, this.props.data.allTypeDocs)}
                onChange={this.onSearchChange}
              />
            </Column>
          </Grid>
        </Layout>

        {!this.state.searchActive && (
          <Layout
            position={Position.Relative}
            zIndex={ZIndex.Default}
            className="landing-page__wrapper"
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            padding={{ top: 5 }}
            fullHeight
          >
            <Layout padding={{ y: 5 }}>
              <Grid>
                <Column cols={{ default: 12, sm: 10, md: 8 }}>
                  <Layout padding={{ bottom: 2 }}>
                    <CoreText
                      type={TextType.H1}
                      color={Color.Overlay}
                      className="landing-page__heading"
                    >
                      Design &amp; Build at Twitch
                    </CoreText>
                  </Layout>
                  <CoreText
                    color={Color.Overlay}
                    className="landing-page__tagline"
                  >
                    The goal of design at Twitch is to ensure usability, accessibility, and performance across the Twitch platform while making it easier for designers and developers to build scalable solutions.
                  </CoreText>
                </Column>
              </Grid>
            </Layout>
            <Layout padding={{ bottom: 5 }}>
              <Grid>
                {this.renderMainTile(
                  'Web Platforms',
                  'Guidelines, design resources, and developer documentation for designing and building for web.',
                  withPrefix('/user-interface'),
                  TileThumb.Web,
                )}
                {this.renderMainTile(
                  'Brand & Marketing',
                  'Identity guidelines and assets — including how to use and create our logos, illustrations, voice, and photography.',
                  withPrefix('/brand'),
                  TileThumb.Brand,
                )}
              </Grid>
            </Layout>

            <Layout padding={{ top: 3, bottom: 2 }}>
              <Grid>
                <Column cols={8}>
                  <CoreText
                    type={TextType.H2}
                    color={Color.Overlay}
                    className="landing-page__sub-heading"
                  >
                    Resources
                  </CoreText>
                </Column>
              </Grid>
            </Layout>
            <Layout padding={{ bottom: 5 }}>
              <Grid>
                {this.renderResourceTile(
                  'Core UI Kit',
                  'Link the Sketch library.',
                  '/resources/ui-kit#sketch-core-ui-kit',
                  'https://s3-us-west-2.amazonaws.com/web-design-int-production/assets/images/thumb-sketch.png',
                )}
                {this.renderResourceTile(
                  'Korolev Font',
                  'Download the Brand font stack.',
                  'https://s3-us-west-2.amazonaws.com/web-design-int-production/assets/downloads/korolev.zip',
                  'https://s3-us-west-2.amazonaws.com/web-design-int-production/assets/images/thumb-korolev.png',
                )}
                {this.renderResourceTile(
                  'Figma Team',
                  'Request access to the team.',
                  withPrefix('/resources/ui-kit#figma-team-library'),
                  'https://s3-us-west-2.amazonaws.com/web-design-int-production/assets/images/thumb-figma.png',
                )}
                {this.renderResourceTile(
                  'UI Prototype Tool',
                  'create-react-app + Core UI',
                  'https://git-aws.internal.justin.tv/core-ui/ui-prototype-tool',
                  'https://s3-us-west-2.amazonaws.com/web-design-int-production/assets/images/thumb-react.png',
                )}
                {this.renderResourceTile(
                  'Video Presentations',
                  'Core UI talks and workshops',
                  withPrefix('/resources/videos'),
                  '',
                )}
              </Grid>
            </Layout>

            <Layout padding={{ bottom: 2 }}>
              <Grid>
                <Column cols={8}>
                  <CoreText
                    type={TextType.H2}
                    color={Color.Overlay}
                    className="landing-page__sub-heading"
                  >
                    Support
                  </CoreText>
                </Column>
              </Grid>
            </Layout>
            <Layout>
              <Grid>
                {this.renderResourceTile(
                  '#prime-carbon-support',
                  'Reach out on slack.',
                  'slack://channel?team=T0266V6GF&id=C2BBMF55E',
                  'https://s3-us-west-2.amazonaws.com/web-design-int-production/assets/images/thumb-slack.png',
                )}
                {this.renderResourceTile(
                  'JIRA',
                  'Open a ticket on Jira.',
                  'https://jira.twitch.com/secure/CreateIssue!default.jspa?pid=25800',
                  'https://s3-us-west-2.amazonaws.com/web-design-int-production/assets/images/thumb-jira.png',
                )}
              </Grid>
            </Layout>
          </Layout>
        )}
      </Layout>
    );
  }

  private onSearchChange = (searchTerm: string) => {
    this.setState({ searchActive: searchTerm.length > 0 ? true : false });
  }

  private renderMainTile = (heading: string, body: string, linkTo: string, imageSrc: TileThumb) => {
    return (
      <Column cols={{ default: 12, sm: 6, md: 4 }}>
        <Layout margin={{ bottom: 1 }} fullHeight display={Display.Flex}>
          <TileLink
            linkTo={linkTo}
          >
            <InjectStyledLayout margin={{ bottom: 1 }} color={Color.Link}>
              {tileThumbPaths[imageSrc]}
            </InjectStyledLayout>
            <Layout display={Display.Flex} alignItems={AlignItems.Center} margin={{ bottom: 1 }}>
              <Layout>
                <CoreText type={TextType.H2}>{heading}</CoreText>
              </Layout>
            </Layout>
            <Layout margin={{ bottom: 1 }} >
              <CoreText color={Color.Alt}>{body}</CoreText>
            </Layout>
          </TileLink>
        </Layout>
      </Column>
    );
  }

  private renderResourceTile = (heading: string, body: string, linkTo: string, imageSrc: string) => {
    return (
      <Column cols={{ default: 12, sm: 4 }}>
        <Layout margin={{ bottom: 1 }}>
          <TileLink
            linkTo={linkTo}
          >
            <Layout display={Display.Flex} alignItems={AlignItems.Center}>
              <Layout padding={{ right: 2 }}>
                <Avatar size={36} alt="" src={imageSrc} />
              </Layout>
              <Layout overflow={Overflow.Hidden}>
                <CoreText type={TextType.H4}>{heading}</CoreText>
                <CoreText ellipsis>{body}</CoreText>
              </Layout>
            </Layout>
          </TileLink>
        </Layout>
      </Column>
    );
  }
}

export const pageQuery = graphql`
  query HomePageContent {
    allPages: allSitePage {
      ...SearchPagesFragment
    }
    allFiles: allFile {
      ...SearchFilesFragment
    }
    allTypeDocs: allTypeDocFile {
      ...SearchTypedocFragment
    }
  }
`;
