import { execSync } from 'child_process';
import * as chokidar from 'chokidar';
import * as crypto from 'crypto';
import * as fs from 'fs';
import { debounce } from 'lodash';
import * as path from 'path';
import * as utils from '../../../utils';
import { TypeDocFile } from '../../graphql-types';

// tslint:disable-next-line
const currentCorePackage = require('carbon-components-prototype/package.json');

const coreRepoURI = currentCorePackage.repository.url;
// Set this to '0.0.0' when in test mode so that snapshots don't thrash on each version bump.
const coreVersion = process.env.NODE_ENV === 'test' ? '0.0.0' : currentCorePackage.version;
const typedocFilePath = path.join(__dirname, '/dist/___typedoc.json');

interface GatsbySourceComponentsArguments {
  componentSrc: string;
}

interface Component {
  children: Property[];
  originalName: string;
  properties: Property[];
}

interface Property {
  availableValues?: AvailableValue[];
  children?: Property[];
  kindString?: string;
  originalName?: string;
  name: string;
  flags: PropertyFlag[];
  comment: Comment;
  breakpointCompatible: boolean;
  inheritedFrom?: {
    name: string;
  };
  type: PropertyType;
  sourceSrc?: string;
}

interface Comment {
  shortText: string;
}

interface PropertyType {
  id?: string;
  name?: string;
  types?: PropertyType[];
  type?: string;
  elementType?: any;
}

interface PropertyFlag {
  isOptional: boolean;
}

interface AvailableValue {
  name: string;
  sourceSrc?: string;
  values?: string[];
  properties?: AvailableValueProperty[];
}

interface AvailableValueProperty {
  name: string;
  value: string;
  isOptional: boolean;
  sourceSrc?: string;
}

/**
 * Create Gatsby nodes for the documentation of each component – watches for
 * changes to component files, recompiles TypeDoc and resources nodes on change.
 */
exports.sourceNodes = async (
  { boundActionCreators, reporter }: any, // tslint:disable-line no-any
  { componentSrc }: GatsbySourceComponentsArguments,
) => {
  const { createNode } = boundActionCreators;
  const watcher = chokidar.watch(componentSrc, {
    ignored: [
      `**/*.scss`,
      `**/*.test.tsx`,
    ],
  });

  const createAndProcessNodes = () => {
    createTypedocFile(componentSrc);

    const fileContents = readTypedocFile();
    const components = filterOnlyComponents(fileContents);

    // Loop through the components.
    components.forEach((component: Component, index: number) => {

      if (component.children) {
        const info = component.children.find((child) => (child.kindString !== undefined && !!child.kindString.match(/(Function|Class)/)));

        if (info) {
          createNode({
            id: `TypeDoc  ${utils.getComponentIdFromPath(component.originalName)}`,
            component: utils.getComponentIdFromPath(component.originalName),
            name: info.name,
            comment: info.comment,
            properties: getProperties(component, fileContents),
            parent: null,
            children: [],
            internal: {
              type: 'TypeDocFile',
              contentDigest: crypto
                .createHash('md5')
                .update(JSON.stringify(fileContents))
                .digest('hex'),
            },
          });
        }

        if (fileContents.children && fileContents.children.length === index + 1) {
          reporter.success('fetch component data');
        }
      }
    });
  };

  createAndProcessNodes();

  watcher.on('change', debounce(() => {
    reporter.info('component file changed – rebuilding typedoc file');
    createAndProcessNodes();
  }, 500));
};

function getProperties (component: Component, fileContents: any) {
  let properties: Property[] = [];

  if (typeof component.children === 'undefined') {
    return properties;
  }

  component.children
    .filter((child) => {
      if (!child.name) {
        return false;
      }
      if (!child.children) {
        return false;
      }
      return child.name.match(/Props$/) && !child.name.match(/BreakpointProps$/);
    })
    .map((child) => {
      if (!child.children) {
        return;
      }
      child.children.map((nextChild) => {
        // Check for duplicates
        if (properties.find((prop) => prop.name === nextChild.name)) {
          return;
        }

        let availableValues: AvailableValue[] = [];
        const firstLevelValues = getFirstLevelValues(nextChild, fileContents);
        const secondLevelValues = getSecondLevelValues(nextChild, fileContents);

        if (firstLevelValues) {
          availableValues = availableValues.concat(firstLevelValues);
        }
        if (secondLevelValues) {
          availableValues = availableValues.concat(secondLevelValues);
        }

        // TODO: Consolidate properties.availableValues and properties.type and properties.sourceSrc
        const sourceRef = findTypedocItem(nextChild.type.id, fileContents);

        // Push the property.
        properties.push({
          name: nextChild.name,
          availableValues,
          flags: nextChild.flags,
          comment: nextChild.comment,
          breakpointCompatible: !!(nextChild.inheritedFrom && nextChild.inheritedFrom.name.match(/BreakpointProps/)),
          type: getType(nextChild),
          sourceSrc: sourceRef && getSrcToSource(sourceRef.sources, coreRepoURI, coreVersion),
        });

      });
    });

  return properties;
}

function getFirstLevelValues (child: Property, fileContents: TypeDocFile): AvailableValue[] {
  let id = child.type.id;
  let typeIsArray = false;

  // If it's an array of another type
  if (!id && child.type.type === 'array') {
    id = child.type.elementType.id;
    typeIsArray = true;
  }

  const ref = findTypedocItem(id, fileContents);
  let value;

  if (ref && ref.children) {
    value = buildValue(ref.id, fileContents);
  } else if (ref && ref.type && ref.type.types && ref.type.types.filter((i: any) => i.type === 'unknown').length > 0) {
    value = buildValue(ref.id, fileContents);
  }

  // If it's an array we'll just re-name the type to be clear
  if (value && typeIsArray) {
    value.name = value.name + '[]';
  }

  if (value) {
    return [value];
  }
  return [];
}

function getSecondLevelValues (child: Property, fileContents: TypeDocFile): AvailableValue[] {
  const ref = findTypedocItem(child.type.id, fileContents);
  const nestedTypesWithIDS = child.type.types && child.type.types.filter((t) => t.id !== undefined);
  let returnValue: AvailableValue[] = [];

  if (ref && ref.type && ref.type.types) {
    ref.type.types.forEach((item: any) => {
      if (item.type === 'unknown') {
        return;
      }
      const value = buildValue(item.id, fileContents);
      if (value) {
        returnValue.push(value);
      }
    });
  } else if (nestedTypesWithIDS) {
    nestedTypesWithIDS.forEach((data) => {
      if (data.id === undefined) {
        return;
      }
      const value = buildValue(data.id, fileContents);
      if (value) {
        returnValue.push(value);
      }
    });
  }

  return returnValue;
}

function buildValue (id: string, fileContents: any): AvailableValue | undefined {
  const ref = findTypedocItem(id, fileContents);

  if (!ref) {
    return;
  }

  // Handles enumerations
  if (ref.kindString === 'Enumeration') {
    return {
      name: ref.name,
      sourceSrc: getSrcToSource(ref.sources, coreRepoURI, coreVersion),
      values: ref.children.map((item: any) => item.name),
    };
  }

  // Handles Enumeration Members
  if (ref.kindString === 'Enumeration member') {
    const refParent = findTypedocParent(id, fileContents);
    return {
      name: refParent && refParent.name,
      sourceSrc: getSrcToSource(ref.sources, coreRepoURI, coreVersion),
      values: [ref.name],
    };
  }

  // Handles nested types
  if (ref.type && ref.type.types) {
    return {
      name: ref.name,
      sourceSrc: getSrcToSource(ref.sources, coreRepoURI, coreVersion),
      values: ref.type.types.map((item: any) => (item.name || item.value)),
    };
  }

  // Handles nested children
  if (ref.children) {
    return {
      name: ref.name,
      sourceSrc: getSrcToSource(ref.sources, coreRepoURI, coreVersion),
      properties: ref.children.map((item: any): AvailableValueProperty => {
        // This may be a reference to another defined type or interface somewhere, or an array of a defined type!
        const refId = (item.type && item.type.id) || (item.type && item.type.type === 'array' && item.type.elementType && item.type.elementType.id);
        const itemRef = findTypedocItem(refId, fileContents);
        const sourceSrc = itemRef && itemRef.sources && getSrcToSource(itemRef.sources, coreRepoURI, coreVersion);

        return {
          name: item.name,
          value: getType(item),
          isOptional: !!item.flags.isOptional,
          sourceSrc,
        };
      }),
    };
  }
}

function getType (child: any) {
  let type;
  let typeIsArray = false;

  // If it's an array of another type
  if (child.type.type === 'array') {
    typeIsArray = true;
  }

  const referenceData = child.type.types && child.type.types.filter((t: any) => t.type === 'reference');
  if (referenceData && referenceData.length > 0) {
    type = referenceData.reduce((accumulator: string, item: any, index: number) => {
      if (item.name !== undefined) {
        return index > 0 ? `${accumulator} | ${item.name}` : item.name;
      } else {
        return accumulator;
      }
    }, '');
  }

  if (child.type.types && child.type.types.find((t: any) => t.name === 'string')) {
    type = 'string';
  } else if (child.type.types && child.type.types.find((t: any) => t.name === 'number')) {
    type = 'number';
  } else if (child.type.types && child.type.types.find((t: any) => t.name && t.name.match(/(true|false)/))) {
    type = 'boolean';
  } else if (child.type.name) {
    type = child.type.name;
  } else if (child.type.elementType && child.type.elementType.name) {
    type = child.type.elementType.name;
  }

  if (typeIsArray) {
    type = type + '[]';
  }

  return type;
}

/**
 * Returns the node which exactly matches the ID provided
 *
 * @param id - Node ID to search by
 * @param data TypeDoc file data node
 */
function findTypedocItem (id: string | undefined, data: any): { [key: string]: any } | undefined {
  if (data.id === id) {
    return data;
  }

  if (!Array.isArray(data.children)) {
    return undefined;
  }

  for (let i = 0; i < data.children.length; i++) {
    const found = findTypedocItem(id, data.children[i]);
    if (found !== undefined) {
      return found;
    }
  }
}

/**
 * Returns the parent node which contains ID provided as one of it's children
 *
 * @param id - Child node ID to search by
 * @param data TypeDoc file data node
 */
function findTypedocParent (id: string, data: any): { [key: string]: any } | undefined {
  if (!Array.isArray(data.children)) {
    return undefined;
  }

  if (data.children.findIndex((child: any) => child.id === id) > -1) {
    return data;
  }

  for (let i = 0; i < data.children.length; i++) {
    const found = findTypedocParent(id, data.children[i]);
    if (found !== undefined) {
      return found;
    }
  }

  return undefined;
}

function getSrcToSource (sources: any[], repoUrl: string, tagVersion: string) {
  if (sources.length === 0) {
    return;
  }
  if (!sources[0].fileName) {
    return;
  }
  if (!sources[0].line) {
    return;
  }
  if (!repoUrl) {
    return;
  }

  const fileName = sources[0].fileName;
  const line = sources[0].line;
  const urlBase = repoUrl.replace(':', '/').replace('.git', '/').replace('git@', 'https://');   // Expects format: git@git-aws.internal.justin.tv:core-ui/core-ui.git
  const branch = tagVersion ? `blob/v${tagVersion}/` : 'blob/master/';
  const dir = `src/`;

  return urlBase + branch + dir + fileName + '#L' + line;
}

function createTypedocFile (sourceDir: any) {
  // Note: Order of arguments is important.
  execSync(`typedoc --json ${typedocFilePath} ${sourceDir} --exclude "**/{*{index\.ts,test\.tsx},/tests/*}" --excludeExternals --ignoreCompilerErrors `);
}

function readTypedocFile () {
  if (fs.existsSync(typedocFilePath)) {
    const fileContents = JSON.parse(fs.readFileSync(typedocFilePath, 'utf8'));
    return fileContents;
  }
}

function filterOnlyComponents (fileContents: any) {
  if (!fileContents || typeof fileContents.children === 'undefined') {
    return [];
  }
  return fileContents.children.filter((item: any) => {
    return (item.name.search('"components/') === 0);
  });
}

module.exports.getSrcToSource = getSrcToSource;
module.exports.getProperties = getProperties;
module.exports.createTypedocFile = createTypedocFile;
module.exports.readTypedocFile = readTypedocFile;
module.exports.filterOnlyComponents = filterOnlyComponents;

/**
 * Resources:
 *  - https://github.com/gatsbyjs/gatsby/blob/master/packages/gatsby-source-filesystem/src/gatsby-node.js
 */
