const plugin = require('../gatsby-node');
const config = require('../../../../gatsby-config');

beforeAll(() => {
  plugin.createTypedocFile(config.plugins.find(item => item.resolve.includes('gatsby-source-components')).options.componentSrc);
});

describe('gatsby source components plugin', () => {

  it('getProperties matches snapshot test', () => {
    const data = plugin.readTypedocFile();
    const components = plugin.filterOnlyComponents(data);

    components.forEach((component) => {
      const result = plugin.getProperties(component, data);
      // The version number which appears in source URLs should be set to 0.0.0 when in test mode in order to reduce snapshot thrashing
      expect(result).toMatchSnapshot(component.name);
    });

  });

  it ('sets the right type for Unions of defined types', () => {
    const data = plugin.readTypedocFile();

    const columnData = data.children.find(child => child.name === '"components/grid/column/component"');
    expect(columnData).not.toBe(undefined);

    const result = plugin.getProperties(columnData, data);
    const colsResult = result.find(item => item.name === 'cols');
    expect(colsResult.type).toBe('ColumnValues | Columns');
  });

  it('sets the right type and values for Unions of enum values', () => {
    const data = plugin.readTypedocFile();

    const columnData = data.children.find(child => child.name === '"components/tabs/tabs/component"');
    expect(columnData).not.toBe(undefined);

    const result = plugin.getProperties(columnData, data);
    const justifyResult = result.find(item => item.name === 'justifyContent');

    expect(justifyResult.type).toBe('Start | Center | End');

    expect(justifyResult.availableValues).toContainEqual({
      name: 'JustifyContent',
      sourceSrc: expect.stringContaining('src/components/layout/component.tsx#L'),
      values: ['Start'],
    });
    expect(justifyResult.availableValues).toContainEqual({
      name: 'JustifyContent',
      sourceSrc: expect.stringContaining('src/components/layout/component.tsx#L'),
      values: ['Center'],
    });
    expect(justifyResult.availableValues).toContainEqual({
      name: 'JustifyContent',
      sourceSrc: expect.stringContaining('src/components/layout/component.tsx#L'),
      values: ['End'],
    });
  });

  it('gets the right values for Layout position', () => {
    const data = plugin.readTypedocFile();

    const layoutData = data.children.find(child => child.name === '"components/layout/component"');
    expect(layoutData).not.toBe(undefined);

    const result = plugin.getProperties(layoutData, data);
    const positionResult = result.find(item => item.name === 'position');

    expect(positionResult.availableValues[0].values).toEqual(['Absolute', 'Fixed', 'Relative']);
  });

  it('gets the right values for Layout padding', () => {
    const data = plugin.readTypedocFile();

    const layoutData = data.children.find(child => child.name === '"components/layout/component"');
    expect(layoutData).not.toBe(undefined);

    const result = plugin.getProperties(layoutData, data);
    const paddingResult = result.find(item => item.name === 'padding');

    expect(paddingResult.availableValues).toContainEqual(expect.objectContaining({
      name: 'PaddingValue',
      values: ['0','0.5','1','2','3','4','5'],
    }));

    expect(paddingResult.availableValues).toContainEqual(expect.objectContaining({
      name:'PaddingValues',
      properties:[
          {
            isOptional:true,
            name:'bottom',
            value:'PaddingValue',
            sourceSrc: expect.stringContaining('src/components/layout/component.tsx#L'),
         },
         {
            isOptional:true,
            name:'left',
            value:'PaddingValue',
            sourceSrc: expect.stringContaining('src/components/layout/component.tsx#L'),
         },
         {
            isOptional:true,
            name:'right',
            value:'PaddingValue',
            sourceSrc: expect.stringContaining('src/components/layout/component.tsx#L'),
         },
         {
            isOptional:true,
            name:'top',
            value:'PaddingValue',
            sourceSrc: expect.stringContaining('src/components/layout/component.tsx#L'),
         },
         {
            isOptional:true,
            name:'x',
            value:'PaddingValue',
            sourceSrc: expect.stringContaining('src/components/layout/component.tsx#L'),
         },
         {
            isOptional:true,
            name:'y',
            value:'PaddingValue',
            sourceSrc: expect.stringContaining('src/components/layout/component.tsx#L'),
         },
      ],
   }));
  });

  it('gets the right values for Column cols', () => {
    const data = plugin.readTypedocFile();

    const columnData = data.children.find(child => child.name === '"components/grid/column/component"');
    expect(columnData).not.toBe(undefined);

    const result = plugin.getProperties(columnData, data);
    const colsResult = result.find(item => item.name === 'cols');

    expect(colsResult.availableValues).toContainEqual(expect.objectContaining({
      name: 'Columns',
      values: ['1','2','3','4','5','6','7','8','9','10','11','12'],
    }));

    expect(colsResult.availableValues).toContainEqual(expect.objectContaining({
      name: 'ColumnValues',
      properties: [
        {
          isOptional: false,
          name: 'default',
          value: 'Columns',
          sourceSrc: expect.stringContaining('src/components/grid/column/component.tsx#L'),
        },
        {
          isOptional: true,
          name: 'lg',
          value: 'Columns',
          sourceSrc: expect.stringContaining('src/components/grid/column/component.tsx#L'),
        },
        {
          isOptional: true,
          name: 'md',
          value: 'Columns',
          sourceSrc: expect.stringContaining('src/components/grid/column/component.tsx#L'),
        },
        {
          isOptional: true,
          name: 'sm',
          value: 'Columns',
          sourceSrc: expect.stringContaining('src/components/grid/column/component.tsx#L'),
        },
        {
          isOptional: true,
          name: 'xl',
          value: 'Columns',
          sourceSrc: expect.stringContaining('src/components/grid/column/component.tsx#L'),
        },
        {
          isOptional: true,
          name: 'xs',
          value: 'Columns',
          sourceSrc: expect.stringContaining('src/components/grid/column/component.tsx#L'),
        },
        {
          isOptional: true,
          name: 'xxl',
          value: 'Columns',
          sourceSrc: expect.stringContaining('src/components/grid/column/component.tsx#L'),
        },
      ],
    }));

  });

  it('gets the right values for complex props like AccordionHeader', () => {
    const data = plugin.readTypedocFile();

    const accordionData = data.children.find(child => child.name === '"components/accordion/accordion-header/component"');
    expect(accordionData).not.toBe(undefined);

    const result = plugin.getProperties(accordionData, data);

    const imageProps = result.find(item => item.name === 'imageProps');

    expect(imageProps.availableValues[0].name).toEqual('CoreImageProps');

    const props = imageProps.availableValues[0].properties;
    expect(props).toContainEqual({ name: 'alt', value: 'string', isOptional: false });
    expect(props).toContainEqual({ name: 'className', value: 'string', isOptional: true });
    expect(props).toContainEqual({ name: 'onError', value: 'React.EventHandler', isOptional: true });
    expect(props).toContainEqual({ name: 'onLoad', value: 'React.EventHandler', isOptional: true });
    expect(props).toContainEqual({ name: 'refHandler', value: undefined, isOptional: true });
    expect(props).toContainEqual({ name: 'sizes', value: 'CoreImageSize[]', isOptional: true, sourceSrc: expect.stringContaining('src/components/core-image/component.tsx#L') });
    expect(props).toContainEqual({ name: 'src', value: 'string', isOptional: false });
    expect(props).toContainEqual({ name: 'srcSet', value: 'CoreSrcSet', isOptional: true, sourceSrc: expect.stringContaining('src/components/core-image/component.tsx#L') });
  });

  it('sets the url to the source code for top level props', () => {
    const data = plugin.readTypedocFile();
    const layoutData = data.children.find(child => child.name === '"components/layout/component"');
    expect(layoutData).not.toBe(undefined);

    const result = plugin.getProperties(layoutData, data);
    const alignResult = result.find(item => item.name === 'alignItems');

    expect(alignResult.availableValues).toContainEqual(
      expect.objectContaining({
        name: 'AlignItems',
        sourceSrc: expect.stringContaining('src/components/layout/component.tsx#L'),
      })
    );
  });

  it('sets the url to the source code for union prop types', () => {
    const data = plugin.readTypedocFile();
    const layoutData = data.children.find(child => child.name === '"components/layout/component"');
    expect(layoutData).not.toBe(undefined);

    const result = plugin.getProperties(layoutData, data);
    const marginResult = result.find(item => item.name === 'margin');

    expect(marginResult.availableValues).toContainEqual(
      expect.objectContaining({
        name: 'MarginValue',
        sourceSrc: expect.stringContaining('src/components/layout/component.tsx#L'),
      })
    );

    expect(marginResult.availableValues).toContainEqual(
      expect.objectContaining({
        name: 'MarginValues',
        sourceSrc: expect.stringContaining('src/components/layout/component.tsx#L'),
      })
    );
  });

});

describe('getSrcToSource', () => {

  it('formats source URLs correctly', () => {
    const gitUrl = 'git@git-aws.internal.justin.tv:samus/carbon-components-prototype.git';
    const version = '1.2.3';
    const sources = [{
      fileName: 'components/foo.ts',
      line: 123,
    }];

    const result = plugin.getSrcToSource(sources, gitUrl, version);

    expect(result).toEqual('https://git-aws.internal.justin.tv/samus/carbon-components-prototype/blob/v1.2.3/src/components/foo.ts#L123');
  });

  it('defaults to master', () => {
    const gitUrl = 'git@git-aws.internal.justin.tv:samus/carbon-components-prototype.git';
    const sources = [{
      fileName: 'components/foo.ts',
      line: 123,
    }];

    const result = plugin.getSrcToSource(sources, gitUrl);

    expect(result).toEqual('https://git-aws.internal.justin.tv/samus/carbon-components-prototype/blob/master/src/components/foo.ts#L123');
  });

  it('fails gracefully', () => {
    const result = plugin.getSrcToSource(['bad data'], 'bad url');
    expect(result).toEqual(undefined);
  });

});
