import * as crypto from 'crypto';
import * as glob from 'glob';
import * as Octokat from 'octokat';
import {
  dummyCommit,
  GatsySourceGithubArguments,
  GithubCommitItem,
  GithubRepo,
} from './';

exports.sourceNodes = (
  { boundActionCreators, reporter }: any, // tslint:disable-line no-any
  { user, repository, token, rootURL, cwd, pattern }: GatsySourceGithubArguments,
) => {
  const { createNode } = boundActionCreators;
  let octo;
  let repo: GithubRepo | undefined = undefined;
  let logged = false;

  if (!user || !repository || !rootURL || !cwd || !pattern) {
    throw 'gatsby-source-github-enterprise is missing configuration options.';
  }

  // Throw an error (to stop build) if running `gatsby build` without specifically setting
  // GATSBY_ENV variable to `ci`.
  if (!token && process.env.GATSBY_ENV !== 'ci' && process.env.NODE_ENV === 'production') {
    throw 'gatsby-source-github-enterprise is missing a Github personal access token.';
  }

  if (token) {
    octo = new Octokat({ token, rootURL });
    repo = octo.repos(user, repository);
  } else {
    reporter.info('could not fetch GitHub data – set your access token');
  }

  glob.sync(`${pattern}`, { cwd }).forEach((path) => {

    // Gatsby needs at least one data point in order to build the GraphQL
    // structure; and without it, the GQL queries will fail on the front-end.
    // To get around this issue when this plugin fails to create real data
    // (due to a missing token or connection issues), we supply a "dummy" node.
    //
    // Note: the site will not build for deployment wihout real data.
    createNode(getProcessedNode(path, [dummyCommit]));

    if (!repo) {
      return;
    }

    // For each component, create a GithubFile node.
    repo.commits.fetch({ path }).then((result: GithubCommitItem) => {

      if (!logged) {
        reporter.success('fetch GitHub data');
        logged = true;
      }

      // Create the GithubFile node for the component.
      createNode(getProcessedNode(path, result.items || []));

    }).catch((error) => {

      // If run during build, stop the build.
      if (process.env.NODE_ENV === 'production') {
        throw error;
      }

      if (!logged) {
        reporter.info(`could not fetch GitHub data – ${error.message}`);
        logged = true;
      }

    });
  });

  return;
};

const getProcessedNode = (component: string, commits: GithubCommitItem[]) => {
  return {
    items: commits,
    id: `Github ${component}`,
    component: component.replace('src/', '').replace(/\/$/, ''),
    parent: null,
    children: [],
    internal: {
      type: 'GithubFile',
      contentDigest: crypto
        .createHash('md5')
        .update(JSON.stringify(commits))
        .digest('hex'),
    },
  };
};
